//this file is part of eMule
//Copyright (C)2002 Merkur ( merkur-@users.sourceforge.net / http://www.emule-project.net )
//
//This program is free software; you can redistribute it and/or
//modify it under the terms of the GNU General Public License
//as published by the Free Software Foundation; either
//version 2 of the License, or (at your option) any later version.
//
//This program is distributed in the hope that it will be useful,
//but WITHOUT ANY WARRANTY; without even the implied warranty of
//MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//GNU General Public License for more details.
//
//You should have received a copy of the GNU General Public License
//along with this program; if not, write to the Free Software
//Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.


#include "StdAfx.h"
#include "serverlist.h"
#include "emule.h"
#include "HttpDownloadDlg.h"
#include "safefile.h"

CServerList::CServerList(CPreferences* in_prefs){
	servercount = 0;
	version = 0;
	serverpos = 0;
	searchserverpos = 0;
	statserverpos = 0;
	app_prefs = in_prefs;
	udp_timer = 0;
	delservercount = 0;
}

void CServerList::AutoUpdate(){
	if (app_prefs->adresses_list.IsEmpty()){
		MessageBox(0,GetResString(IDS_ERR_EMPTYADRESSESDAT),GetResString(IDS_ERR_EMPTYADRESSESDAT_TITLE),64);
		return;
	}
	bool bDownloaded=false;
	char* servermet = new char[strlen(app_prefs->GetAppDir())+20];
	char* oldservermet = new char[strlen(app_prefs->GetAppDir())+20];
	CString strURLToDownload; 
	sprintf(servermet,"%sserver.met",app_prefs->GetAppDir());
	sprintf(oldservermet,"%sserver_met.old",app_prefs->GetAppDir());
	remove(oldservermet);
	rename(servermet,oldservermet);
	
	POSITION Pos = app_prefs->adresses_list.GetHeadPosition(); 
	while ((!bDownloaded) && (Pos != NULL)){
		CHttpDownloadDlg dlgDownload;
		strURLToDownload = app_prefs->adresses_list.GetNext(Pos); 
		dlgDownload.m_sURLToDownload = strURLToDownload.GetBuffer();
		dlgDownload.m_sFileToDownloadInto = servermet;
		if (dlgDownload.DoModal() == IDOK){
			remove(oldservermet);
			bDownloaded=true;
		}
		else{
			theApp.emuledlg->AddLogLine(true,GetResString(IDS_ERR_FAILEDDOWNLOADMET), strURLToDownload.GetBuffer());
		}
	}
	if (!bDownloaded)
		rename(oldservermet,servermet);
	delete[] servermet;
	delete[] oldservermet;
}

bool CServerList::Init(){
	// auto update the list by using an url
	if (app_prefs->AutoServerlist())
		AutoUpdate();
	// Load Metfile
	CString strPath = app_prefs->GetAppDir() + CString("server.met");
	bool bRes = AddServermetToList(strPath, false);	// this deletes all existing servers
	// insert static servers from textfile
	strPath = app_prefs->GetAppDir() + CString("staticservers.dat");
	AddServersFromTextFile(strPath);
	return bRes;
}

bool CServerList::AddServermetToList(CString strFile, bool merge){
	if (!merge)
	{
		theApp.emuledlg->serverwnd.serverlistctrl.DeleteAllItems();
		this->RemoveAllServers();
	}
	CSafeFile servermet;
	try{
		if (!servermet.Open(strFile,CFile::modeRead|CFile::osSequentialScan)){
			theApp.emuledlg->AddLogLine(false,GetResString(IDS_ERR_LOADSERVERMET));
			return false;
		}
		servermet.Read(&version,1);
		if (version != 0xE0 && version != MET_HEADER){
			servermet.Close();
			theApp.emuledlg->AddLogLine(false,GetResString(IDS_ERR_BADSERVERMETVERSION),version);
			return false;
		}
		uint32 fservercount;
		servermet.Read(&fservercount,4);
		
		ServerMet_Struct sbuffer;
		uint32 iAddCount = 0;
		for (uint32 j = 0;j != fservercount;j++){
			// get server
			servermet.Read(&sbuffer,sizeof(ServerMet_Struct));
			CServer* newserver = new CServer(&sbuffer);
			//add tags
			for (uint32 i=0;i != sbuffer.tagcount;i++)
				newserver->AddTagFromFile(&servermet);
			// set listname for server
			if (!newserver->GetListName()){
				char* listname = new char[strlen(newserver->GetAddress())+8];
				sprintf(listname,"Server %s",newserver->GetAddress());
				newserver->SetListName(listname);
				delete listname;
			}
			if (!theApp.emuledlg->serverwnd.serverlistctrl.AddServer(newserver,true) ){
				CServer* update = theApp.serverlist->GetServerByAddress(newserver->GetAddress(), newserver->GetPort());
				if(update){
					update->SetListName( newserver->GetListName());
					if( newserver->GetDescription() )
						update->SetDescription( newserver->GetDescription());
					theApp.emuledlg->serverwnd.serverlistctrl.RefreshServer(update);
				}
				delete newserver;
			}
			else
				iAddCount++;
		}

		if (!merge)
			theApp.emuledlg->AddLogLine(true,GetResString(IDS_SERVERSFOUND),fservercount);
		else
			theApp.emuledlg->AddLogLine(true,GetResString(IDS_SERVERSADDED), iAddCount, fservercount-iAddCount);
		servermet.Close();
	}
	catch(CFileException* error){
		OUTPUT_DEBUG_TRACE();
		if (error->m_cause == CFileException::endOfFile)
			theApp.emuledlg->AddLogLine(true,GetResString(IDS_ERR_BADSERVERLIST));
		else{
			char buffer[150];
			error->GetErrorMessage(buffer,150);
			theApp.emuledlg->AddLogLine(true,GetResString(IDS_ERR_FILEERROR_SERVERMET),buffer);
		}
		error->Delete();	//memleak fix
	}
	return true;
}

bool CServerList::AddServer(CServer* in_server){
	if (theApp.glob_prefs->FilerBadIPs()) 
		if ( !IsGoodServerIP( in_server ))
			return false;
	CServer* test_server = GetServerByAddress(in_server->GetAddress(), in_server->GetPort());
	if (test_server){
		test_server->ResetFailedCount();
		theApp.emuledlg->serverwnd.serverlistctrl.RefreshServer( test_server );		
		return false;
	}
	list.AddTail(in_server);
	return true;
}

void CServerList::ServerStats(){
	if( theApp.serverconnect->IsConnected() && list.GetCount() > 0 ) {
		CServer* ping_server = GetNextStatServer();
		CServer* test = ping_server;
		if( !ping_server )
			return;
        while(ping_server->GetLastPinged() != 0 && (::GetTickCount() - ping_server->GetLastPinged()) < UDPSERVSTATREASKTIME){ 
			ping_server = this->GetNextStatServer();
			if( ping_server == test )
				return;
		}
		if( ping_server->GetFailedCount() >= theApp.glob_prefs->GetDeadserverRetries() && theApp.glob_prefs->DeadServer() ){
			theApp.emuledlg->serverwnd.serverlistctrl.RemoveServer(ping_server);
			return;
		}
		Packet* packet = new Packet( OP_GLOBSERVSTATREQ, 4 );
		srand((unsigned)time(NULL));
		uint32 time = 0x55AA0000 + (uint16)rand();
		ping_server->SetChallenge(time);
		memcpy( packet->pBuffer, &time, 4 );
		ping_server->SetLastPinged( ::GetTickCount() );
		ping_server->AddFailedCount();
		theApp.emuledlg->serverwnd.serverlistctrl.RefreshServer( ping_server );
		theApp.serverconnect->SendUDPPacket( packet, ping_server, true );
	}
}

/*filter Servers with invalid IP's / Port
0.*
10.*
172.16.0.0 - 172.31.255.255
192.168.0.0 - 192.168.255.255
127.*
*/
bool CServerList::IsGoodServerIP(CServer* in_server){ 
	if (in_server->HasDynIP())
		return true;
	char * pch;

	char ipx[24];
	sprintf(ipx,"%s",in_server->GetFullIP());
	
	// Determine first and second number if the given IP
	pch = strtok (ipx,".");	CString first=pch;pch=strtok(NULL,".");	CString second=pch;

	int secondI=atoi(second);

	if (first=="172" && (secondI>=16 && secondI<=31) ) return false;

	if (first=="192" && secondI==168 ) return false;

	if (first=="0" || first=="10" || first=="127"
		|| in_server->GetPort()==0) return false;

	return true; 
}

void CServerList::RemoveServer(CServer* out_server){
	for(POSITION pos = list.GetHeadPosition(); pos != NULL; )
	{
		POSITION pos2 = pos;
		CServer* test_server = list.GetNext(pos);
		if (test_server == out_server){
			if (theApp.downloadqueue->cur_udpserver == out_server)
				theApp.downloadqueue->cur_udpserver = 0;
			list.RemoveAt(pos2);
			delservercount++;
			delete test_server;
			return;
		}
	}
}

void CServerList::RemoveAllServers(){
	for(POSITION pos = list.GetHeadPosition(); pos != NULL; pos = list.GetHeadPosition()) {
		delete list.GetAt(pos);
		list.RemoveAt(pos);
		delservercount++;
	}
}

void CServerList::GetStatus(uint32 &total, uint32 &failed, uint32 &user, uint32 &file, uint32 &tuser, uint32 &tfile,float &occ){
	total = list.GetCount();
	failed = 0;
	user = 0;
	file = 0;
	tuser=0;
	tfile = 0;
	occ=0;
	uint32 maxusers=0;
	uint32 tuserk = 0;

	CServer* curr;
	for (POSITION pos = list.GetHeadPosition(); pos !=0;list.GetNext(pos)){
		curr = (CServer*)list.GetAt(pos);
		if( curr->GetFailedCount() ){
			failed++;
		}
		else{
			user += curr->GetUsers();
			file += curr->GetFiles();
		}
		tuser += curr->GetUsers();
		tfile += curr->GetFiles();
		
		if (curr->GetMaxUsers()) {
			tuserk += curr->GetUsers(); // total users on servers with known maximum
			maxusers+=curr->GetMaxUsers();
		}
	}
	if (maxusers>0) occ=(float)(tuserk*100)/maxusers;
}

void CServerList::GetUserFileStatus(uint32 &user, uint32 &file){
	user = 0;
	file = 0;
	CServer* curr;
	for (POSITION pos = list.GetHeadPosition(); pos !=0;list.GetNext(pos)){
		curr = (CServer*)list.GetAt(pos);
		if( !curr->GetFailedCount() ){
			user += curr->GetUsers();
			file += curr->GetFiles();
		}
	}
}

CServerList::~CServerList(){
	SaveServermetToFile();
	for(POSITION pos = list.GetHeadPosition(); pos != NULL; pos = list.GetHeadPosition()) {
		delete list.GetAt(pos);
		list.RemoveAt(pos);
	}
	if (udp_timer)
		KillTimer(0,udp_timer);
}

void CServerList::MoveServerDown(CServer* aServer){
   POSITION pos1, pos2;
   uint16 i = 0;
   for( pos1 = list.GetHeadPosition(); ( pos2 = pos1 ) != NULL; ){
	   list.GetNext(pos1);
	   CServer* cur_server = list.GetAt(pos2);
	   if (cur_server==aServer){
		   list.AddTail(cur_server);
		   list.RemoveAt(pos2);
		   return;
	   }
	   i++;
	   if (i == list.GetCount())
		   break;
   }
}

void CServerList::Sort(){
   POSITION pos1, pos2;
   uint16 i = 0;
   for( pos1 = list.GetHeadPosition(); ( pos2 = pos1 ) != NULL; ){
	   list.GetNext(pos1);
	   CServer* cur_server = list.GetAt(pos2);
	   if (cur_server->GetPreferences()== PR_HIGH){
		   list.AddHead(cur_server);
		   list.RemoveAt(pos2);
	   }
	   else if (cur_server->GetPreferences() == PR_LOW){
		   list.AddTail(cur_server);
		   list.RemoveAt(pos2);
	   }
	   i++;
	   if (i == list.GetCount())
		   break;
   }
}

CServer* CServerList::GetNextServer(){
	CServer* nextserver = 0;
	uint32 i = 0;
	if (serverpos>=((uint32)list.GetCount()) ) return 0;
	while (!nextserver && i != list.GetCount()){
		POSITION posIndex = list.FindIndex(serverpos);
		if (posIndex == NULL) {	// check if search position is still valid (could be corrupted by server delete operation)
			posIndex = list.GetHeadPosition();
			serverpos= 0;       //<<--9/27/02 zg
		}

		nextserver = list.GetAt(posIndex);
		serverpos++;
		i++;
		// TODO: Add more option to filter bad server like min ping, min users etc
		//if (nextserver->preferences = ?)
		//	nextserver = 0;
		//if (serverpos == list.GetCount()) return 0;//			serverpos = 0;
	}
	return nextserver;
}

CServer* CServerList::GetNextSearchServer(){
	CServer* nextserver = 0;
	uint32 i = 0;
	while (!nextserver && i != list.GetCount()){
		POSITION posIndex = list.FindIndex(searchserverpos);
		if (posIndex == NULL) {	// check if search position is still valid (could be corrupted by server delete operation)
			posIndex = list.GetHeadPosition();
			searchserverpos=0;
		}
		nextserver = list.GetAt(posIndex);
		searchserverpos++;
		i++;
		if (searchserverpos == list.GetCount())
			searchserverpos = 0;
	}
	return nextserver;
}

CServer* CServerList::GetNextStatServer(){
	CServer* nextserver = 0;
	uint32 i = 0;
	while (!nextserver && i != list.GetCount()){
		POSITION posIndex = list.FindIndex(statserverpos);
		if (posIndex == NULL) {	// check if search position is still valid (could be corrupted by server delete operation)
			posIndex = list.GetHeadPosition();
			statserverpos=0;
		}

		nextserver = list.GetAt(posIndex);
		statserverpos++;
		i++;
		if (statserverpos == list.GetCount())
			statserverpos = 0;
	}
	return nextserver;
}

bool CServerList::BroadCastPacket(Packet* packet){ // unused atm . but might be useful later
	if (udp_timer)
		return false;
	udp_timer = SetTimer(0,4322,UDPSEARCHSPEED,CServerList::UDPTimerProc);
	broadcastpos = list.GetHeadPosition();
	broadcastpacket = packet;
	return true;
}

void CServerList::SendNextPacket(){
	if (theApp.listensocket->TooManySockets()){
		KillTimer(0,udp_timer);
		udp_timer = 0;
		delete broadcastpacket;
		return;
	}

	if (broadcastpos != 0){
		CServer* cur_server = list.GetAt(broadcastpos);
		if (cur_server != theApp.serverconnect->GetCurrentServer())
			theApp.serverconnect->SendUDPPacket(broadcastpacket,cur_server,false);
		list.GetNext(broadcastpos);
	}
	else{
		KillTimer(0,udp_timer);
		udp_timer = 0;
		delete broadcastpacket;

	}
}

void CServerList::CancelUDPBroadcast(){
	if (udp_timer){
		KillTimer(0,udp_timer);
		udp_timer = 0;
		delete broadcastpacket;
	}
}

void CALLBACK CServerList::UDPTimerProc(HWND hwnd, UINT uMsg,UINT_PTR idEvent,DWORD dwTime){
	theApp.serverlist->SendNextPacket();
}

CServer* CServerList::GetNextServer(CServer* lastserver){
	if (list.IsEmpty())
		return 0;
	if (!lastserver) return list.GetHead();

	POSITION pos = list.Find(lastserver);
	if (!pos){
		TRACE("Error: CServerList::GetNextServer");
		return list.GetHead();
	}
	list.GetNext(pos);
	if (!pos)
		return NULL;
	else
		return list.GetAt(pos);
}

CServer* CServerList::GetServerByAddress(char* address, uint16 port){
	for (POSITION pos = list.GetHeadPosition();pos != 0;list.GetNext(pos)){
        CServer *s = list.GetAt(pos);   // i_a: small speed optimization 
        if (port == s->GetPort() && !strcmp(s->GetAddress(),address)) 
          return s; 
	}
	return NULL;
}

bool CServerList::SaveServermetToFile(){
	char* newservermet = new char[strlen(app_prefs->GetAppDir())+MAX_PATH];
	sprintf(newservermet,"%sserver.met.new",app_prefs->GetAppDir());
//	theApp.emuledlg->AddLogLine( false, "%s", newservermet );
	try{
	FILE* servermet = fopen(newservermet, "wb");
	if (!servermet){
		theApp.emuledlg->AddLogLine(false,GetResString(IDS_ERR_SAVESERVERMET));
		delete[] newservermet;	//mf
		return false;
	}
	version = 0xE0;

	fwrite( &version, 1, 1, servermet);
	uint32 fservercount = list.GetCount();
	fwrite(&fservercount,4,1,servermet);
	ServerMet_Struct sbuffer;
	CServer* nextserver;

	for (uint32 j = 0;j != fservercount;j++){
		nextserver = this->GetServerAt(j);
		sbuffer.ip = nextserver->GetIP();
		sbuffer.port = nextserver->GetPort();
		uint16 tagcount = 7;
		if (nextserver->GetListName())
			tagcount++;
		if (nextserver->GetDynIP())
			tagcount++;
		if (nextserver->GetDescription())
			tagcount++;
		sbuffer.tagcount = tagcount;
		fwrite( &sbuffer, 1, sizeof(ServerMet_Struct),servermet);
		
		if( nextserver->GetListName() ){
			CTag servername( ST_SERVERNAME, nextserver->GetListName() );
			servername.WriteTagToFile(servermet);
		}
		if( nextserver->GetDynIP() ){
			CTag serverdynip( ST_DYNIP, nextserver->GetDynIP() );
			serverdynip.WriteTagToFile(servermet);
		}
		if( nextserver->GetDescription() ){
			CTag serverdesc( ST_DESCRIPTION, nextserver->GetDescription() );
			serverdesc.WriteTagToFile(servermet);
		}
		CTag serverfail(ST_FAIL, nextserver->GetFailedCount() );
		serverfail.WriteTagToFile(servermet);
		CTag serverpref( ST_PREFERENCE, nextserver->GetPreferences() );
		serverpref.WriteTagToFile(servermet);
		CTag serveruser("users", nextserver->GetUsers() );
		serveruser.WriteTagToFile(servermet);
		CTag serverfiles("files", nextserver->GetFiles() );
		serverfiles.WriteTagToFile(servermet);
		CTag serverping(ST_PING, nextserver->GetPing() );
		serverping.WriteTagToFile(servermet);
		CTag serverlastp(ST_LASTPING, nextserver->GetLastPinged() );
		serverlastp.WriteTagToFile(servermet);
		CTag servermaxusers(ST_MAXUSERS, nextserver->GetMaxUsers() );
		servermaxusers.WriteTagToFile(servermet);
	}
	fclose(servermet);
	char* curservermet = new char[strlen(app_prefs->GetAppDir())+20];
	char* oldservermet = new char[strlen(app_prefs->GetAppDir())+20];
	sprintf(curservermet,"%sserver.met",app_prefs->GetAppDir());
	sprintf(oldservermet,"%sserver_met.old",app_prefs->GetAppDir());
	remove(oldservermet);
	rename(curservermet,oldservermet);
	rename(newservermet,curservermet);
	delete[] oldservermet;
	delete[] curservermet;
	}
	catch(CFileException* error) {
		OUTPUT_DEBUG_TRACE();
		theApp.emuledlg->AddLogLine(false,GetResString(IDS_ERR_SAVESERVERMET2));
		delete[] newservermet;	//mf
		error->Delete();
		return false;
	}
	delete[] newservermet;
	return true;
}

void CServerList::AddServersFromTextFile(CString strFilename,bool isstaticserver) {
	CString strLine;
	CStdioFile f;
	if (!f.Open(strFilename, CFile::modeRead | CFile::typeText))
		return;
	while(f.ReadString(strLine)) {
		// format is host:port,Name
		if (strLine.GetLength() < 5)
			continue;
		if (strLine.GetAt(0) == '#' || strLine.GetAt(0) == '/')
			continue;

		// fetch host
		int pos = strLine.Find(':');
		if (pos == -1){
			pos = strLine.Find(','); 
			if (pos == -1) 
			continue;
		}
		CString strHost = strLine.Left(pos);
		strLine = strLine.Mid(pos+1);
		// fetch  port
		pos = strLine.Find(',');
		if (pos == -1)
			continue;
		CString strPort = strLine.Left(pos);
		strLine = strLine.Mid(pos+1);
		// fetch name
		CString strName = strLine;
		strName.Replace("\r", "");
		strName.Replace("\n", "");

		// create server object and add it to the list
		CServer* nsrv = new CServer(atoi(strPort), strHost.GetBuffer());
		nsrv->SetListName(strName.GetBuffer());
		nsrv->SetIsStaticMember(isstaticserver);
		nsrv->SetPreference(2);
		if (!theApp.emuledlg->serverwnd.serverlistctrl.AddServer(nsrv, true))
		{
			delete nsrv;
			CServer* srvexisting = GetServerByAddress(strHost.GetBuffer(), atoi(strPort));
			if (srvexisting) {
				srvexisting->SetListName(strName.GetBuffer());
				srvexisting->SetIsStaticMember(isstaticserver);
				srvexisting->SetPreference(2);
				if (theApp.emuledlg->serverwnd) theApp.emuledlg->serverwnd.serverlistctrl.RefreshServer(srvexisting);
			}

		}
	}

	f.Close();
}
