//Copyright (C)2002 Merkur ( merkur-@users.sourceforge.net / http://www.emule-project.net )
//
//This program is free software; you can redistribute it and/or
//modify it under the terms of the GNU General Public License
//as published by the Free Software Foundation; either
//version 2 of the License, or (at your option) any later version.
//
//This program is distributed in the hope that it will be useful,
//but WITHOUT ANY WARRANTY; without even the implied warranty of
//MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//GNU General Public License for more details.
//
//You should have received a copy of the GNU General Public License
//along with this program; if not, write to the Free Software
//Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

#include "StdAfx.h"
#include "partfile.h"
#include "emule.h"
#include "updownclient.h"
#include <math.h>
#include "ED2KLink.h"
#include "Preview.h"
#include "ini2.h"

CBarShader CPartFile::s_LoadBar(5); 
CBarShader CPartFile::s_ChunkBar(16); 

CPartFile::CPartFile(){
	Init();
}

CPartFile::CPartFile(CSearchFile* searchresult){
	Init();
	memcpy(filehash,searchresult->GetFileHash(),16);
	for (int i = 0; i != searchresult->taglist.GetCount();i++){
		switch (searchresult->taglist[i]->tag->specialtag){
			case FT_FILENAME:{
				filename = nstrdup(searchresult->taglist[i]->tag->stringvalue);
				break;
			}
			case FT_FILESIZE:{
				filesize = searchresult->taglist[i]->tag->intvalue;
				break;
			}
			default:
				CTag* newtag = new CTag(searchresult->taglist[i]->tag);
				taglist.Add(newtag);
		}
	}
	CreatePartFile();
}

CPartFile::CPartFile(CString edonkeylink)
{
	CED2KLink* pLink = 0;
	try {
		pLink = CED2KLink::CreateLinkFromUrl(edonkeylink);
		_ASSERT( pLink != 0 );
		CED2KFileLink* pFileLink = pLink->GetFileLink();
		if (pFileLink==0) 
			throw GetResString(IDS_ERR_NOTAFILELINK);
		InitializeFromLink(pFileLink);
	} catch (CString error) {
		OUTPUT_DEBUG_TRACE();
		char buffer[200];
		sprintf(buffer,GetResString(IDS_ERR_INVALIDLINK),error.GetBuffer());
		theApp.emuledlg->AddLogLine(true, GetResString(IDS_ERR_LINKERROR), buffer);
		status = PS_ERROR;
	}
	delete pLink;
}

void
CPartFile::InitializeFromLink(CED2KFileLink* fileLink)
{
	Init();
	try{
		filename = nstrdup( fileLink->GetName() );
		filesize = fileLink->GetSize();
		memcpy(filehash,fileLink->GetHashKey(),sizeof(filehash));
		if (!theApp.downloadqueue->IsFileExisting(filehash))
			CreatePartFile();
		else
			status = PS_ERROR;
		}
	catch(CString error){
		OUTPUT_DEBUG_TRACE();
		char buffer[200];
		sprintf(buffer,GetResString(IDS_ERR_INVALIDLINK),error.GetBuffer());
		theApp.emuledlg->AddLogLine(true, GetResString(IDS_ERR_LINKERROR), buffer);
		status = PS_ERROR;
	}
}

CPartFile::CPartFile(CED2KFileLink* fileLink)
{
	InitializeFromLink(fileLink);
}

void CPartFile::Init(){
	fullname = 0;
	newdate = true;
	lastsearchtime = 0;
	paused = false;
	status = PS_EMPTY;
	transfered = 0;
	priority = PR_NORMAL;
	srcarevisible = false;
	transferingsrc = 0;
	datarate = 0;
	hashsetneeded = true;
	count = 0;
	percentcompleted = 0;
	partmetfilename = 0;
	completedsize=0;
	m_bPreviewing = false;
	lastseencomplete=NULL;
	availablePartsCount=0;
	m_ClientSrcAnswered = 0;
	m_LastNoNeededCheck = 0;
	m_bAutoPriority = false ; 
	m_iRate=0;
	m_strComment="";
}

CPartFile::~CPartFile(){
	if (fullname)
		delete[] fullname;
	if (partmetfilename)
		delete[] partmetfilename;
	m_SrcpartFrequency.RemoveAll();
	for (POSITION pos = gaplist.GetHeadPosition();pos != 0;gaplist.GetNext(pos))
		delete gaplist.GetAt(pos);
}

void CPartFile::CreatePartFile(){
	// use lowest free partfilenumber for free file (InterCeptor)
	int i = 0; 
	CString filename; 
	do 
	{ 
		i++; 
		filename.Format("%s\\%03i.part", theApp.glob_prefs->GetTempDir(), i); 
	
	}while(PathFileExists(filename)); 
	partmetfilename = new char[15]; 
	sprintf(partmetfilename,"%03i.part.met",i); 

	fullname = new char[strlen(theApp.glob_prefs->GetTempDir())+strlen(partmetfilename)+MAX_PATH];
	sprintf(fullname,"%s\\%s",theApp.glob_prefs->GetTempDir(),partmetfilename);
	char* buffer = nstrdup(partmetfilename);
	buffer[strlen(buffer)-4] = 0;
	CTag* partnametag = new CTag(FT_PARTFILENAME,buffer);
	delete[] buffer;
	taglist.Add(partnametag);
	
	Gap_Struct* gap = new Gap_Struct;
	gap->start = 0;
	gap->end = filesize-1;
	gaplist.AddTail(gap);

	char* partfull = nstrdup(fullname);
	partfull[strlen(partfull)-4] = 0;
	
	
	if (!m_hpartfile.Open(partfull,CFile::modeCreate|CFile::modeReadWrite|CFile::shareDenyWrite|CFile::osSequentialScan)){
		theApp.emuledlg->AddLogLine(false,GetResString(IDS_ERR_CREATEPARTFILE));
		status = PS_ERROR;
	}
	delete[] partfull;
	m_SrcpartFrequency.SetSize(GetPartCount());
	for (uint32 i = 0; i != GetPartCount();i++)
		m_SrcpartFrequency.Add(0);
	paused = false;
	//UAP Hunter
	if (theApp.glob_prefs->IsUAPEnabled())
		((CKnownFile*)this)->SetAutoPriority(true);
	else
		((CKnownFile*)this)->SetAutoPriority(false);

	((CKnownFile*)this)->UpdateUploadAutoPriority();
	//end UAP
	SavePartFile();
}

bool CPartFile::LoadPartFile(char* in_directory,char* in_filename){
	partmetfilename = nstrdup(in_filename);
	directory = nstrdup(in_directory);
	char* buffer = new char[strlen(directory)+strlen(partmetfilename)+2];
	sprintf(buffer,"%s\\%s",directory,partmetfilename);
	fullname = buffer;
	CSafeFile file;
	try{
		// readfile data form part.met file
		if (!file.Open(fullname,CFile::modeRead)){
			theApp.emuledlg->AddLogLine(false,GetResString(IDS_ERR_OPENMET),partmetfilename,filename);
			return false;
		}
		uint8 version;
		file.Read(&version,1);
		if (version != PARTFILE_VERSION){
			if (file!= INVALID_HANDLE_VALUE) file.Close();
			theApp.emuledlg->AddLogLine(false,GetResString(IDS_ERR_BADMETVERSION),partmetfilename,filename);
			return false;
		}
		LoadDateFromFile(&file);
		LoadHashsetFromFile(&file,false);

		uint32 tagcount;
		file.Read(&tagcount,4);
		for (uint32 j = 0; j != tagcount;j++){
			CTag* newtag = new CTag(&file);
			switch(newtag->tag->specialtag){
				case FT_FILENAME:{
					if(newtag->tag->stringvalue == NULL) {
						theApp.emuledlg->AddLogLine(true,GetResString(IDS_ERR_METCORRUPT),partmetfilename,filename);
						delete newtag;
						return false;
					}
					filename = nstrdup(newtag->tag->stringvalue);
					delete newtag;
					break;
				}
				case FT_LASTSEENCOMPLETE: {
					lastseencomplete = newtag->tag->intvalue;
					delete newtag;
					break;
				}
				case FT_FILESIZE:{
					filesize = newtag->tag->intvalue;
					delete newtag;
					break;
				}
				case FT_TRANSFERED:{
					transfered = newtag->tag->intvalue;
					delete newtag;
					break;
				}
				case FT_PRIORITY:{
					priority = newtag->tag->intvalue;
					if ((priority==PR_AUTO)&&(!theApp.glob_prefs->IsUAPEnabled())) priority=PR_NORMAL;
					delete newtag;
					if (priority == PR_AUTO)
						this->SetAutoPriority(true);
					else
						this->SetAutoPriority(false);
					break;
				}
				case FT_STATUS:{
					paused = newtag->tag->intvalue;
					delete newtag;
					break;
				}
				case FT_ULPRIORITY:{ 
					if ((priority==PR_AUTO)&&(!theApp.glob_prefs->IsUAPEnabled())) priority=PR_NORMAL;
					if ((uint8)newtag->tag->intvalue == PR_AUTO) {
						((CKnownFile*)this)->SetAutoPriority(true);
						((CKnownFile*)this)->UpdateUploadAutoPriority();
					} else {
						((CKnownFile*)this)->SetAutoPriority(false);
						((CKnownFile*)this)->SetPriority((uint8)newtag->tag->intvalue);
					}
					delete newtag;
					break; 
				}
				default:{
					if ((!newtag->tag->specialtag) && newtag->tag->tagname[0] == FT_GAPSTART){					
						Gap_Struct* gap = new Gap_Struct; 
						gap->start = newtag->tag->intvalue;
						delete newtag;
						j++;
						CTag* newtag = new CTag(&file);
						if (newtag->tag->tagname[0] != FT_GAPEND){
							delete newtag;
							throw GetResString(IDS_ERR_BADGAP);
						}
						gap->end = newtag->tag->intvalue;
						// gap start = first missing byte but gap ends = first non-missing byte in edonkey
						// but I think its easier to user the real limits so:
						gap->end--;
						gaplist.AddTail(gap);
						delete newtag;
					}
					else
						taglist.Add(newtag);
				}
					
			}

		}
		if (file != INVALID_HANDLE_VALUE) file.Close();
	}
	catch(CFileException* error){
		OUTPUT_DEBUG_TRACE();
		if (error->m_cause == CFileException::endOfFile)
			theApp.emuledlg->AddLogLine(true,GetResString(IDS_ERR_METCORRUPT),partmetfilename,filename);
		else{
			char buffer[150];
			error->GetErrorMessage(buffer,150);
			theApp.emuledlg->AddLogLine(true,GetResString(IDS_ERR_FILEERROR),partmetfilename,filename,error);
		}
		error->Delete();	//memleak fix
		return false;
	}

	//check if this is a backup
	if(stricmp(strrchr(fullname, '.'), ".backup") == 0) {
		char *shorten = strrchr(fullname, '.');
		*shorten = 0;
		fullname = (char*)realloc(fullname, strlen(fullname) + 1);
	}

	// open permanent handle
	char* searchpath = nstrdup(fullname);
	searchpath[strlen(fullname)-4] = 0;
	if (!m_hpartfile.Open(searchpath,CFile::modeReadWrite|CFile::shareDenyWrite|CFile::osSequentialScan)){
		theApp.emuledlg->AddLogLine(false,GetResString(IDS_ERR_FILEOPEN),fullname,filename);
		delete[] searchpath;
		return false;
	}
	delete[] searchpath;
			

	m_SrcpartFrequency.SetSize(GetPartCount());
	for (uint32 i = 0; i != GetPartCount();i++)
		m_SrcpartFrequency.Add(0);
	status = PS_READY;
	// check hashcount, filesatus etc
	if (hashlist.GetCount() < GetPartCount()){
		status = PS_EMPTY;
		hashsetneeded = true;
		return true;
	}
	else {
		hashsetneeded = false;
		bool empty = true;
		for (int i = 0; i != hashlist.GetSize(); i++){
			if (IsComplete(i*PARTSIZE,((i+1)*PARTSIZE)-1))
				empty = false;
		}
		if (empty)
			status = PS_EMPTY;
	}

	if (gaplist.IsEmpty()){	// is this file complete already?
		CompleteFile(false);
		return true;
	}

	// check date of .part file - if its wrong, rehash file
		CFileStatus filestatus;
		m_hpartfile.GetStatus(filestatus);
		if (date != mktime(filestatus.m_mtime.GetLocalTm())){
			theApp.emuledlg->AddLogLine(false,GetResString(IDS_ERR_REHASH),buffer,filename);
			// rehash
			status = PS_WAITINGFORHASH;
			CAddFileThread* addfilethread = (CAddFileThread*) AfxBeginThread(RUNTIME_CLASS(CAddFileThread), THREAD_PRIORITY_NORMAL,0, CREATE_SUSPENDED);
			addfilethread->SetValues(0,directory,m_hpartfile.GetFileName().GetBuffer(),this);
			addfilethread->ResumeThread();	
		}

	return true;
}

bool CPartFile::SavePartFile(){
	switch (status){
		case PS_WAITINGFORHASH:
		case PS_HASHING:
			return false;
	}
	FILE* file = 0;
	try{
		//get filedate
		CFileFind ff;
		char* searchpath = nstrdup(fullname);
		searchpath[strlen(fullname)-4] = 0;
		bool end = !ff.FindFile(searchpath,0);
		delete[] searchpath;
		if (!end)
				ff.FindNextFile();
		if (end || ff.IsDirectory())
			throw GetResString(IDS_ERR_PART_FNF);
		CTime lwtime;
		ff.GetLastWriteTime(lwtime);
		date = mktime(lwtime.GetLocalTm());
		ff.Close();
		uint32 lsc = mktime(lastseencomplete.GetLocalTm());
		CString backupName(fullname);
		backupName.Append(".backup");

		remove(backupName);
		rename(fullname, backupName);

		// readfile data form part.met file
		file = fopen(fullname,"wbS");
		if (!file)
			throw GetResString(IDS_ERR_OPENMETFILE);
		//version
		uint8 version = PARTFILE_VERSION;
		fwrite(&version,1,1,file);
		//date
		fwrite(&date,4,1,file);
		//hash
		fwrite(&filehash,16,1,file);
		uint16 parts = hashlist.GetCount();
		fwrite(&parts,2,1,file);
		for (int x = 0; x != parts; x++)
			fwrite(hashlist[x],16,1,file);
		//tags
		uint32 tagcount = taglist.GetCount()+7+(gaplist.GetCount()*2);
		fwrite(&tagcount,4,1,file);
		CTag* nametag = new CTag(FT_FILENAME,filename);
		nametag->WriteTagToFile(file);
		delete nametag;
		CTag* sizetag = new CTag(FT_FILESIZE,filesize);
		sizetag->WriteTagToFile(file);
		delete sizetag;
		CTag* transtag = new CTag(FT_TRANSFERED,transfered);
		transtag->WriteTagToFile(file);
		delete transtag;
		CTag* statustag = new CTag(FT_STATUS,(paused)? 1:0);
		statustag->WriteTagToFile(file);
		delete statustag;
		//UAP Hunter
		CTag* prioritytag ;
		if (IsAutoPrioritized()) {
			uint8 aux_priority = PR_AUTO ;
			prioritytag = new CTag(FT_PRIORITY,aux_priority);
		}
		else
			prioritytag = new CTag(FT_PRIORITY,priority);
		prioritytag->WriteTagToFile(file);
		delete prioritytag;
		//end UAP
		CTag* lsctag = new CTag(FT_LASTSEENCOMPLETE,lsc);
		lsctag->WriteTagToFile(file);
		delete lsctag;

		// Modified by Tarod for UAP
		CTag* ulprioritytag;
		if (((CKnownFile*)this)->IsAutoPrioritized())
			ulprioritytag = new CTag(FT_ULPRIORITY, PR_AUTO);
		else
			ulprioritytag = new CTag(FT_ULPRIORITY,m_iPriority);
		ulprioritytag->WriteTagToFile(file);
		delete ulprioritytag;
		// End by Tarod
		for (uint32 j = 0; j != taglist.GetCount();j++)
			taglist[j]->WriteTagToFile(file);
		//gaps
		char* namebuffer = new char[10];
		char* number = &namebuffer[1];
		uint16 i_pos = 0;
		for (POSITION pos = gaplist.GetHeadPosition();pos != 0;gaplist.GetNext(pos)){
			itoa(i_pos,number,10);
			namebuffer[0] = FT_GAPSTART;
			CTag* gapstarttag = new CTag(namebuffer,gaplist.GetAt(pos)->start);
			gapstarttag->WriteTagToFile(file);
			// gap start = first missing byte but gap ends = first non-missing byte in edonkey
			// but I think its easier to user the real limits
			namebuffer[0] = FT_GAPEND;
			CTag* gapendtag = new CTag(namebuffer,(gaplist.GetAt(pos)->end)+1);
			gapendtag->WriteTagToFile(file);
			delete gapstarttag;
			delete gapendtag;
			i_pos++;
		}
		delete[] namebuffer;

		if (ferror(file))
			throw CString("unexpected write error");

		remove(backupName);
	}
	catch(char* error){
		OUTPUT_DEBUG_TRACE();
		if (file)
			fclose(file);
		theApp.emuledlg->AddLogLine(false,GetResString(IDS_ERR_SAVEMET),error,partmetfilename,filename);
		return false;
	}
	catch(CString error){
		OUTPUT_DEBUG_TRACE();
		if (file)
			fclose(file);
		theApp.emuledlg->AddLogLine(false,GetResString(IDS_ERR_SAVEMET),error.GetBuffer(),partmetfilename,filename);
		return false;
	}
	fclose(file);
	return true;
}

void CPartFile::PartFileHashFinished(CKnownFile* result){
	newdate = true;
	bool errorfound = false;
	for (uint32 i = 0; i != hashlist.GetSize(); i++){
		if (IsComplete(i*PARTSIZE,((i+1)*PARTSIZE)-1)){
			if (!(result->GetPartHash(i) && !memcmp(result->GetPartHash(i),this->GetPartHash(i),16))){
				theApp.emuledlg->AddLogLine(false,GetResString(IDS_ERR_FOUNDCORRUPTION),i+1,filename);		
				AddGap(i*PARTSIZE,((((i+1)*PARTSIZE)-1) >= filesize) ? filesize-1 : ((i+1)*PARTSIZE)-1);
				errorfound = true;
			}
		}
	}
	delete result;
	if (!errorfound){
		if (status == PS_COMPLETING){
			CompleteFile(true);
			return;
		}
		else
			theApp.emuledlg->AddLogLine(false,GetResString(IDS_HASHINGDONE),filename);
	}
	else{
		status = PS_READY;
		SavePartFile();
		return;
	}
	status = PS_READY;
	SavePartFile();
	theApp.sharedfiles->SafeAddKFile(this);
}

void CPartFile::AddGap(uint32 start, uint32 end){
	POSITION pos1, pos2;
	for (pos1 = gaplist.GetHeadPosition();(pos2 = pos1) != NULL;){
		Gap_Struct* cur_gap = gaplist.GetNext(pos1);
		if (cur_gap->start >= start && cur_gap->end <= end){ // this gap is inside the new gap - delete
			gaplist.RemoveAt(pos2);
			delete cur_gap;
			continue;
		}
		else if (cur_gap->start >= start && cur_gap->start <= end){// a part of this gap is in the new gap - extend limit and delete
			end = cur_gap->end;
			gaplist.RemoveAt(pos2);
			delete cur_gap;
			continue;
		}
		else if (cur_gap->end <= end && cur_gap->end >= start){// a part of this gap is in the new gap - extend limit and delete
			start = cur_gap->start;
			gaplist.RemoveAt(pos2);
			delete cur_gap;
			continue;
		}
		else if (start >= cur_gap->start && end <= cur_gap->end){// new gap is already inside this gap - return
			return;
		}
	}
	Gap_Struct* new_gap = new Gap_Struct;
	new_gap->start = start;
	new_gap->end = end;
	gaplist.AddTail(new_gap);
	theApp.emuledlg->transferwnd.downloadlistctrl.UpdateItem(this);
	newdate = true;
}

bool CPartFile::IsComplete(uint32 start, uint32 end){
	if (end >= filesize)
		end = filesize-1;
	for (POSITION pos = gaplist.GetHeadPosition();pos != 0;gaplist.GetNext(pos)){
		Gap_Struct* cur_gap = gaplist.GetAt(pos);
		if ((cur_gap->start >= start && cur_gap->end <= end)||(cur_gap->start >= start 
			&& cur_gap->start <= end)||(cur_gap->end <= end && cur_gap->end >= start)
			||(start >= cur_gap->start && end <= cur_gap->end)){
				return false;	
			}
	}
	return true;
}

bool CPartFile::IsPureGap(uint32 start, uint32 end){
	if (end >= filesize)
		end = filesize-1;
	for (POSITION pos = gaplist.GetHeadPosition();pos != 0;gaplist.GetNext(pos)){
		Gap_Struct* cur_gap = gaplist.GetAt(pos);
		if (start >= cur_gap->start  && end <= cur_gap->end ){
			return true;
		}
	}
	return false;
}

bool CPartFile::IsAlreadyRequested(uint32 start, uint32 end){
	for (POSITION pos =  requestedblocks_list.GetHeadPosition();pos != 0; requestedblocks_list.GetNext(pos)){
		Requested_Block_Struct* cur_block =  requestedblocks_list.GetAt(pos);
		if (cur_block->StartOffset == start && cur_block->EndOffset == end)
			return true;
	}
	return false;
}

bool CPartFile::GetNextEmptyBlockInPart(uint16 partnumber,Requested_Block_Struct* result){
	for (uint32 i = 0; i != ceil((float)PARTSIZE/BLOCKSIZE);i++){
		uint32 start = (PARTSIZE*partnumber) + i*BLOCKSIZE;
		uint32 end	 = (PARTSIZE*partnumber) + ((i+1)*BLOCKSIZE)-1;
		if (end >= (uint32)PARTSIZE*(partnumber+1))
			end = (PARTSIZE*(partnumber+1))-1;
		if (start >= GetFileSize())
			break;
		if (end >= GetFileSize())
			end = GetFileSize()-1;
		if ( (!IsComplete(start,end)) && !IsAlreadyRequested(start,end)){
			if (result){
				result->StartOffset = start;
				result->EndOffset = end;
				memcpy(result->FileID,GetFileHash(),16);
			}
			return true;
		}
	}
	return false;
}

void CPartFile::FillGap(uint32 start, uint32 end){
	POSITION pos1, pos2;
	for (pos1 = gaplist.GetHeadPosition();(pos2 = pos1) != NULL;){
		Gap_Struct* cur_gap = gaplist.GetNext(pos1);
		if (cur_gap->start >= start && cur_gap->end <= end){ // our part fills this gap completly
			gaplist.RemoveAt(pos2);
			delete cur_gap;
			continue;
		}
		else if (cur_gap->start >= start && cur_gap->start <= end){// a part of this gap is in the part - set limit
			cur_gap->start = end+1;
		}
		else if (cur_gap->end <= end && cur_gap->end >= start){// a part of this gap is in the part - set limit
			cur_gap->end = start-1;
		}
		else if (start >= cur_gap->start && end <= cur_gap->end){
			uint32 buffer = cur_gap->end;
			cur_gap->end = start-1;
			cur_gap = new Gap_Struct;
			cur_gap->start = end+1;
			cur_gap->end = buffer;
			gaplist.InsertAfter(pos1,cur_gap);
			break; // [Lord KiRon]
		}
	}

	UpdateCompletedInfos();
	theApp.emuledlg->transferwnd.downloadlistctrl.UpdateItem(this);
	newdate = true;
	SavePartFile();
}

void CPartFile::UpdateCompletedInfos() {

   	uint32 allgaps = 0; 

	for (POSITION pos = gaplist.GetHeadPosition();pos !=  0;gaplist.GetNext(pos)){ 
		Gap_Struct* cur_gap = gaplist.GetAt(pos); 
		allgaps += cur_gap->end - cur_gap->start;
	}

	if ((gaplist.GetCount() || requestedblocks_list.GetCount()))
	{ 
			percentcompleted = ((1.0f-(float)allgaps/filesize)) * 100; 
			completedsize = (uint32)((filesize-(allgaps+1))); 
	} else {
		percentcompleted = 100; completedsize=filesize;
	}
}

void CPartFile::DrawStatusBar(CDC* dc, RECT* rect, bool bFlat){ 
	COLORREF crProgress;
	COLORREF crHave;
	COLORREF crPending;
	COLORREF crMissing = RGB(255, 0, 0);

	if(bFlat) {
		crProgress = RGB(0, 150, 0);
		crHave = RGB(0, 0, 0);
		crPending = RGB(255,255,100);
	} else {
		crProgress = RGB(0, 224, 0);
		crHave = RGB(104, 104, 104);
		crPending = RGB(255, 208, 0);
	}

	s_ChunkBar.SetHeight(rect->bottom - rect->top);
	s_ChunkBar.SetWidth(rect->right - rect->left);
	s_ChunkBar.SetFileSize(filesize);
	s_ChunkBar.Fill(crHave);

	uint32 allgaps = 0;

	if(status == PS_COMPLETE || status == PS_COMPLETING) {
		s_ChunkBar.FillRange(0, filesize, crProgress);
		s_ChunkBar.Draw(dc, rect->left, rect->top, bFlat);
		percentcompleted = 100; completedsize=filesize;
		return;
	}

	// red gaps
	for (POSITION pos = gaplist.GetHeadPosition();pos !=  0;gaplist.GetNext(pos)){
		Gap_Struct* cur_gap = gaplist.GetAt(pos);
		allgaps += cur_gap->end - cur_gap->start;
		bool gapdone = false;
		uint32 gapstart = cur_gap->start;
		uint32 gapend = cur_gap->end;
		for (uint32 i = 0; i != GetPartCount(); i++){
			if (gapstart >= i*PARTSIZE && gapstart <=  (i+1)*PARTSIZE){ // is in this part?
				if (gapend <= (i+1)*PARTSIZE)
					gapdone = true;
				else{
					gapend = (i+1)*PARTSIZE; // and next part
				}
				// paint
				COLORREF color;
				if (m_SrcpartFrequency.GetCount() >= (INT_PTR)i && m_SrcpartFrequency[i])  // frequency?
					color = RGB(0,
					(210-(22*(m_SrcpartFrequency[i]-1)) <  0)? 0:210-(22*(m_SrcpartFrequency[i]-1))
					,255);
				else
					color = crMissing;

				s_ChunkBar.FillRange(gapstart, gapend + 1,  color);

				if (gapdone) // finished?
					break;
				else{
					gapstart = gapend;
					gapend = cur_gap->end;
				}
			}
		}
	}

	// yellow pending parts
	for (POSITION pos = requestedblocks_list.GetHeadPosition();pos !=  0;requestedblocks_list.GetNext(pos)){
		Requested_Block_Struct* block =  requestedblocks_list.GetAt(pos);
		s_ChunkBar.FillRange(block->StartOffset, block->EndOffset,  crPending);
	}

	s_ChunkBar.Draw(dc, rect->left, rect->top, bFlat);

	// green progress
	float blockpixel = (float)(rect->right -  rect->left)/((float)filesize);
	RECT gaprect;
	gaprect.top = rect->top;
	gaprect.bottom = gaprect.top + 4;
	gaprect.left = rect->left;

	if(!bFlat) {
		s_LoadBar.SetWidth((uint32)((float)((float)((filesize-((allgaps==0)?1:allgaps) )-1))*blockpixel + .5f));
		s_LoadBar.Fill(crProgress);
		s_LoadBar.Draw(dc, gaprect.left, gaprect.top, false);
	} else {
		gaprect.right = rect->left+  (uint32)((float)((float)((filesize-((allgaps==0)?1:allgaps))-1))*blockpixel +  .5f);
		dc->FillRect(&gaprect, &CBrush(crProgress));
		//draw gray progress only if flat
		gaprect.left = gaprect.right;
		gaprect.right = rect->right;
		dc->FillRect(&gaprect, &CBrush(RGB(224,224,224)));
	}

	if (	   (gaplist.GetCount() || requestedblocks_list.GetCount())	   )
	{
		percentcompleted = ((1.0f-(float)allgaps/filesize)) * 100;
		completedsize = (uint32)((filesize-(allgaps+1)));
	} else {percentcompleted = 100; completedsize=filesize;}

}

void CPartFile::WritePartStatus(CFile* file){
	uint16 parts = hashlist.GetCount();
	file->Write(&parts,2);
	uint16 done = 0;
	while (done != parts){
		uint8 towrite = 0;
		for (uint32 i = 0;i != 8;i++){
			if (IsComplete(done*PARTSIZE,((done+1)*PARTSIZE)-1))
				towrite |= (1<<i);
			done++;
			if (done == parts)
				break;
		}
		file->Write(&towrite,1);
	}
}

int CPartFile::GetValidSourcesCount() {
	int counter=0;
	POSITION pos1,pos2;
	for (pos1 = srclist.GetHeadPosition();( pos2 = pos1 ) != NULL;){
		srclist.GetNext(pos1);
		CUpDownClient* cur_src = srclist.GetAt(pos2);
		if (cur_src->GetDownloadState()!=DS_ONQUEUE && cur_src->GetDownloadState()!=DS_DOWNLOADING &&
			cur_src->GetDownloadState()!=DS_NONEEDEDPARTS) counter++;
	}
	return counter;
}

uint16 CPartFile::GetNotCurrentSourcesCount(){
		uint16 counter=0;

		POSITION pos1,pos2;
		for (pos1 = srclist.GetHeadPosition();( pos2 = pos1 ) != NULL;){
			srclist.GetNext(pos1);
			CUpDownClient* cur_src = srclist.GetAt(pos2);
			if (cur_src->GetDownloadState()!=DS_ONQUEUE && cur_src->GetDownloadState()!=DS_DOWNLOADING) counter++;
		}
		return counter;
}

uint8 CPartFile::GetStatus(bool ignorepause){
	if ((!paused) || status == PS_ERROR || ignorepause)
		return status;
	else
		return PS_PAUSED;
}

uint32 CPartFile::Process(sint16 reducedownload/*in percent*/){
	// check if we want new sources from server
	uint16 test = theApp.glob_prefs->GetMaxSourcePerFileSoft();
	if (( (!lastsearchtime) || (::GetTickCount() - lastsearchtime) > SERVERREASKTIME) && theApp.serverconnect->IsConnected()
		&& theApp.glob_prefs->GetMaxSourcePerFileSoft() > GetSourceCount() ){
		//local server
		lastsearchtime = ::GetTickCount();
		Packet* packet = new Packet(OP_GETSOURCES,16);
		memcpy(packet->pBuffer,filehash,16);
		theApp.serverconnect->SendPacket(packet,true);
	}

	transferingsrc = 0;
	datarate = 0;
	POSITION pos1,pos2;
	for (pos1 = srclist.GetHeadPosition();( pos2 = pos1 ) != NULL;){
		srclist.GetNext(pos1);

		CUpDownClient* cur_src = srclist.GetAt(pos2);

			switch (cur_src->GetDownloadState()){
					case DS_DOWNLOADING:{
						transferingsrc++;
						uint32 cur_datarate = cur_src->CalculateDownloadRate();
						datarate += cur_datarate;
						if (reducedownload && cur_src->GetDownloadState() == DS_DOWNLOADING){
							uint32 limit = (uint32)(((float)reducedownload/100)*cur_datarate)/10;		
								if (limit < 1000 && reducedownload == 200)
									limit += 1000;
								else if (limit < 1)
									limit = 1;
								cur_src->socket->SetDownloadLimit(limit);
						}
						else
							cur_src->socket->DisableDownloadLimit();
						break;
					}
					case DS_BANNED:
					case DS_ERROR:
						break;
					case DS_LOWTOLOWIP:	// if we now have a high ip we can ask
						if (theApp.serverconnect->IsLowID())
							break;
					case DS_NONEEDEDPARTS:{ // doubled reasktime for no needed parts - save connections and traffic
						if (!((!cur_src->GetLastAskedTime()) || (::GetTickCount() - cur_src->GetLastAskedTime()) > FILEREASKTIME*2))
							break;
					}
					case DS_ONQUEUE:{
						if (theApp.serverconnect->IsConnected() && ((!cur_src->GetLastAskedTime()) || (::GetTickCount() - cur_src->GetLastAskedTime()) > FILEREASKTIME-20000)){
							cur_src->UDPReaskForDownload();
						}
					}
					case DS_CONNECTING:
					case DS_TOOMANYCONNS:
					case DS_CONNECTED:
					case DS_NONE:
					case DS_WAITCALLBACK:
						if (theApp.serverconnect->IsConnected() && ((!cur_src->GetLastAskedTime()) || (::GetTickCount() - cur_src->GetLastAskedTime()) > FILEREASKTIME)){
							cur_src->AskForDownload();
						}
						break;
			}
	}


	// swap No needed partfiles if possible
	if (((!m_LastNoNeededCheck) || (::GetTickCount() - m_LastNoNeededCheck) > 10000)){
		m_LastNoNeededCheck = ::GetTickCount();
		POSITION pos1,pos2;
		for (pos1 = srclist.GetHeadPosition();( pos2 = pos1 ) != NULL;){
			srclist.GetNext(pos1);
			CUpDownClient* cur_src = srclist.GetAt(pos2);
			if (cur_src->GetDownloadState() == DS_NONEEDEDPARTS)
				cur_src->SwapToAnotherFile(false);
		}
	}


	// calculate datarate, set limit etc.

	count++;
	if (count == 30){
		count = 0;
		theApp.emuledlg->transferwnd.downloadlistctrl.UpdateItem(this);
	}
	return datarate;
}

void CPartFile::AddSources(CMemFile* sources,uint32 serverip, uint16 serverport){
	uint8 count;
	uint8 debug_lowiddropped = 0;
	uint8 debug_possiblesources = 0;
	sources->Read(&count,1);
	for (int i = 0;i != count;i++){
		uint32 userid;
		sources->Read(&userid,4);
		uint16 port;
		sources->Read(&port,2);
		// check first if we are this source

		// MOD Note: Do not change this part - Merkur
		if (theApp.serverconnect->GetClientID() < 16777216 && theApp.serverconnect->IsConnected()){
			if ((theApp.serverconnect->GetClientID() == userid) && inet_addr(theApp.serverconnect->GetCurrentServer()->GetFullIP()) == serverip)
				continue;
		}
		else if (theApp.serverconnect->GetClientID() == userid)
			continue;
		else if (userid < 16777216 && !theApp.serverconnect->IsLocalServer(serverip,serverport)){
			debug_lowiddropped++;
			continue;
		}
		// MOD Note - end

		if( theApp.glob_prefs->GetMaxSourcePerFile() > this->GetSourceCount() ){
			debug_possiblesources++;
			CUpDownClient* newsource = new CUpDownClient(port,userid,serverip,serverport,this);
			theApp.downloadqueue->CheckAndAddSource(this,newsource);
		}
	}
	DEBUG_ONLY(theApp.emuledlg->AddLogLine(false,CString("Debug: %i sources received, %i low id dropped, %i possible sources"),count,debug_lowiddropped,debug_possiblesources));
}

void CPartFile::NewSrcPartsInfo(){
	if (m_SrcpartFrequency.GetCount() != GetPartCount()){
		m_SrcpartFrequency.RemoveAll();
		m_SrcpartFrequency.SetSize(0);
		for (int i = 0; i != GetPartCount();i++)
			m_SrcpartFrequency.Add(0);
	}
	else{
		for (int i = 0; i != GetPartCount();i++)
			m_SrcpartFrequency[i] = 0;
	}
	for (POSITION pos = srclist.GetHeadPosition();pos != 0;srclist.GetNext(pos)){
		for (int i = 0; i != GetPartCount(); i++){
			if (srclist.GetAt(pos)->IsPartAvailable(i))
				m_SrcpartFrequency[i] +=1;
		}
	}
	theApp.emuledlg->transferwnd.downloadlistctrl.UpdateItem(this);
}

bool CPartFile::GetNextRequestedBlock(CUpDownClient* sender,Requested_Block_Struct** newblocks,uint16* count){
#define	BLOCKCOUNT		3

	uint16 newblockcount = 0;
	uint8* partsav = sender->GetPartStatus();
	*count = 0;
	bool finished = false;
	while (!finished){
		uint16 randomness = (uint16)ROUND(((float)rand()/RAND_MAX)*(GetPartCount()-1));
		CList<int,int> liGoodParts;
		CList<int,int> liPossibleParts;
		for (uint16 i = 0;i != GetPartCount();i++){
			if (partsav[randomness] && !IsComplete(randomness*PARTSIZE,((randomness+1)*PARTSIZE)-1)){
				/*if (IsCorruptedPart(randomness)){
					if (GetNextEmptyBlockInPart(randomness,0)){
						goodpart = randomness;
						break;
					}
				}
				else*/ 
				if (IsMovie() && (randomness <= 1 || randomness >= GetPartCount()-2)){
					// if this is a movie we want the last und first part first to make a preview possible
					if (GetNextEmptyBlockInPart(randomness,0)){
							liGoodParts.AddHead(randomness);	
							break;
					}
				}
				else if (IsPureGap(randomness*PARTSIZE,((randomness+1)*PARTSIZE)-1)){
					if (GetNextEmptyBlockInPart(randomness,0))
						liPossibleParts.AddHead(randomness);
				}
				else{
					if (GetNextEmptyBlockInPart(randomness,0)){
						liGoodParts.AddHead(randomness);
					}
				}
			}
			randomness++;
			if (randomness == GetPartCount())
				randomness = 0;
		}

		CList<int,int>* usedlist;
		if (!liGoodParts.IsEmpty())
			usedlist = &liGoodParts;
		else if (!liPossibleParts.IsEmpty())
			usedlist = &liPossibleParts;
		else{
			if (!newblockcount){
				return false;
			}
			else

				break;
		}
		uint16 nRarest = 0xFFFF;
		uint16 usedpart = usedlist->GetHead();
		for (POSITION pos = usedlist->GetHeadPosition();pos != 0;usedlist->GetNext(pos)){
			if (m_SrcpartFrequency.GetCount() >= usedlist->GetAt(pos)
			&& m_SrcpartFrequency[usedlist->GetAt(pos)] < nRarest){
				nRarest = m_SrcpartFrequency[usedlist->GetAt(pos)];
				usedpart = usedlist->GetAt(pos);
			}
		}

		while (true){
			Requested_Block_Struct* block = new Requested_Block_Struct;
			if (GetNextEmptyBlockInPart(usedpart,block)){
				requestedblocks_list.AddTail(block);
				newblocks[newblockcount] = block;
				newblockcount++;
				if (newblockcount == BLOCKCOUNT){
					finished = true;
					break;
				}
			}
			else
			{
				delete block;
				break;
			}
		}

	} //wend
	*count = newblockcount;
	return true;
}

void  CPartFile::RemoveBlockFromList(uint32 start,uint32 end){
	POSITION pos1,pos2;
	for (pos1 = requestedblocks_list.GetHeadPosition(); ( pos2 = pos1 ) != NULL; ){
	   requestedblocks_list.GetNext(pos1);
		if (requestedblocks_list.GetAt(pos2)->StartOffset <= start && requestedblocks_list.GetAt(pos2)->EndOffset >= end)
			requestedblocks_list.RemoveAt(pos2);
	}
}

void  CPartFile::BlockReceived(uint32 start,uint32 end, char* blockdata,uint32 transferedsize){
	// TODO buffer blocks to decrease hd access
	RemoveBlockFromList(start,end);
	if (IsComplete(start,end))
		return;
	try{

		if (m_hpartfile.GetLength() < end+1)
			m_hpartfile.SetLength(end+1);
		m_hpartfile.Seek(start,0);
		m_hpartfile.Write(blockdata,(end-start)+1);
		uint16 partnumber = (uint16)floor((float)start/PARTSIZE);		
		uint32 length = PARTSIZE;
		if (PARTSIZE*(partnumber+1) > m_hpartfile.GetLength())
			length = (m_hpartfile.GetLength()- (PARTSIZE*partnumber));		
		m_hpartfile.Flush();

		FillGap(start,end);

		if (transferedsize) 
			transfered += transferedsize;
		else
			 transfered += (end-start)+1;

		// if this part is complete or was corrupted check the hash

	
		if (IsComplete(PARTSIZE*partnumber,(PARTSIZE*(partnumber+1))-1)){
			if (!HashSinglePart(partnumber)){
				theApp.emuledlg->AddLogLine(true,GetResString(IDS_ERR_PARTCORRUPT),partnumber,GetFileName());
				AddGap(PARTSIZE*partnumber,(PARTSIZE*partnumber+length)-1);
				corrupted_list.AddTail(partnumber);
			}
			else{
				if (status == PS_EMPTY){
					status = PS_READY;
					theApp.sharedfiles->SafeAddKFile(this);
				}
			}
		}
		else if (IsCorruptedPart(partnumber) && theApp.glob_prefs->IsICHEnabled()){
			if (HashSinglePart(partnumber)){
				FillGap(PARTSIZE*partnumber,(PARTSIZE*partnumber+length)-1);
				RemoveBlockFromList(PARTSIZE*partnumber,(PARTSIZE*partnumber+length)-1);
				theApp.emuledlg->AddLogLine(true,GetResString(IDS_ICHWORKED),partnumber,GetFileName());
			}
		}
		
		if (gaplist.IsEmpty())		//file complete
			CompleteFile(false);	
	}
	catch(CFileException* error){
		OUTPUT_DEBUG_TRACE();
		if(theApp.glob_prefs->IsErrorBeepEnabled()) Beep(800,200);

		if (error->m_cause == CFileException::diskFull) {
			theApp.emuledlg->AddLogLine(true,GetResString(IDS_ERR_OUTOFSPACE),this->GetFileName());
			if (theApp.glob_prefs->GetNotifierPopOnImportantError()) {
				CString msg;
				msg.Format(GetResString(IDS_ERR_OUTOFSPACE),this->GetFileName());
				theApp.emuledlg->ShowNotifier(msg, TBN_IMPORTANTEVENT, false);
			}
		}else{
			char buffer[150];
			error->GetErrorMessage(buffer,150);
			theApp.emuledlg->AddLogLine(true,GetResString(IDS_ERR_WRITEERROR),GetFileName(),buffer);
			status = PS_ERROR;
		}
		paused = true;
		datarate = 0;
		transferingsrc = 0;
	
		theApp.emuledlg->transferwnd.downloadlistctrl.UpdateItem(this);
		error->Delete();
	}
}

void CPartFile::CompleteFile(bool hashingdone){
	status = PS_COMPLETING;
	if (!hashingdone){
		datarate = 0;
		char* partfileb = nstrdup(partmetfilename);
		partfileb[strlen(partmetfilename)-4] = 0;
		CAddFileThread* addfilethread = (CAddFileThread*) AfxBeginThread(RUNTIME_CLASS(CAddFileThread), THREAD_PRIORITY_BELOW_NORMAL,0, CREATE_SUSPENDED);
		addfilethread->SetValues(0,theApp.glob_prefs->GetTempDir(),partfileb,this);
		addfilethread->ResumeThread();	
		delete[] partfileb;
		return;
	}
	else{
		StopFile();
		char* partfilename = nstrdup(fullname);
		partfilename[strlen(fullname)-4] = 0;
		char* newfilename = nstrdup(GetFileName());
		strcpy(newfilename, (LPCTSTR)theApp.StripInvalidFilenameChars(newfilename));
		char* newname = new char[strlen(newfilename)+strlen(theApp.glob_prefs->GetIncomingDir())+MAX_PATH];
		sprintf(newname,"%s\\%s",theApp.glob_prefs->GetIncomingDir(),newfilename);

		// close permanent handle
		if (m_hpartfile != INVALID_HANDLE_VALUE) m_hpartfile.Close();

		bool renamed = false;
		if(PathFileExists(newname)) {
			renamed = true;
			int namecount = 0;

			size_t length = strlen(newfilename);
			ASSERT(length != 0); //name should never be 0

			//the file extension
			char *ext = strrchr(newfilename, '.');
			if(ext == NULL)
				ext = newfilename + length;

			char *last = ext;  //new end is the file name before extension
			last[0] = 0;  //truncate file name

			//serch for matching ()s and check if it contains a number
			if((ext != newfilename) && (strrchr(newfilename, ')') + 1 == last)) {
				char *first = strrchr(newfilename, '(');
				if(first != NULL) {
					first++;
					bool found = true;
					for(char *step = first; step < last - 1; step++)
						if(*step < '0' || *step > '9') {
							found = false;
							break;
						}
					if(found) {
						namecount = atoi(first);
						last = first - 1;
						last[0] = 0;  //truncate again
					}
				}
			}

			CString strTestName;
			do {
				namecount++;
				strTestName.Format("%s\\%s(%d).%s", theApp.glob_prefs->GetIncomingDir(),
					newfilename, namecount, min(ext + 1, newfilename + length));
			} while(PathFileExists(strTestName));
			delete[] newname;
			newname = nstrdup(strTestName);
		}
		delete newfilename;

		if (rename(partfilename,newname)){
			delete[] partfilename;
			delete[] newname;
			theApp.emuledlg->AddLogLine(true,GetResString(IDS_ERR_COMPLETIONFAILED),GetFileName());
			paused = true;
			status = PS_ERROR;
			return;
		}
		if (remove(fullname))
			theApp.emuledlg->AddLogLine(true,GetResString(IDS_ERR_DELETEFAILED),fullname);

		delete[] partfilename;
		delete[] fullname;
		fullname = newname;
		delete[] directory;
		directory = nstrdup(theApp.glob_prefs->GetIncomingDir());
		status = PS_COMPLETE;
		paused = false;
		theApp.emuledlg->AddLogLine(true,GetResString(IDS_DOWNLOADDONE),GetFileName());
		theApp.emuledlg->ShowNotifier(GetResString(IDS_TBN_DOWNLOADDONE)+"\n"+GetFileName(), TBN_DLOAD);
		if (renamed)
			theApp.emuledlg->AddLogLine(true, GetResString(IDS_DOWNLOADRENAMED), strrchr(newname, '\\') + 1);
		theApp.knownfiles->SafeAddKFile(this);
		((CKnownFile*)this)->SetAutoPriority(false) ; //UAP
		((CKnownFile*)this)->SetPriority(PR_NORMAL) ; //UAP
		theApp.downloadqueue->RemoveFile(this);
		theApp.emuledlg->transferwnd.downloadlistctrl.UpdateItem(this);
	}
}

void  CPartFile::RemoveAllSources(bool bTryToSwap){
	//TODO transfer sources to other downloading files if possible
	POSITION pos1,pos2;
	for( pos1 = srclist.GetHeadPosition(); ( pos2 = pos1 ) != NULL; ){
		srclist.GetNext(pos1);
		if (bTryToSwap){
			if (!srclist.GetAt(pos2)->SwapToAnotherFile(true))
				theApp.downloadqueue->RemoveSource(srclist.GetAt(pos2));
		}
		else
			theApp.downloadqueue->RemoveSource(srclist.GetAt(pos2));
	}
}

void CPartFile::DeleteFile(){
	ASSERT ( !m_bPreviewing );

	theApp.sharedfiles->RemoveFile(this);
	theApp.downloadqueue->RemoveFile(this);
	theApp.emuledlg->transferwnd.downloadlistctrl.RemoveFile(this);
	RemoveAllSources(true);

	if (m_hpartfile != INVALID_HANDLE_VALUE) m_hpartfile.Close();

	if (remove(fullname))
		theApp.emuledlg->AddLogLine(true,GetResString(IDS_ERR_DELETE),fullname);
	char* partfilename = nstrdup(fullname);
	partfilename[strlen(fullname)-4] = 0;
	if (remove(partfilename))
		theApp.emuledlg->AddLogLine(true,GetResString(IDS_ERR_DELETE),partfilename);

	delete[] partfilename;
	delete this;
}

bool CPartFile::HashSinglePart(uint16 partnumber){
	if ((GetHashCount() <= partnumber) && (GetPartCount() > 1)){
		theApp.emuledlg->AddLogLine(true,GetResString(IDS_ERR_HASHERRORWARNING),GetFileName());
		return true;
	}
	else if(!GetPartHash(partnumber)){
		theApp.emuledlg->AddLogLine(true,GetResString(IDS_ERR_INCOMPLETEHASH),GetFileName());
		return true;		
	}
	else{
		uchar hashresult[16];
		m_hpartfile.Seek(PARTSIZE*partnumber,0);
		uint32 length = PARTSIZE;
		if (PARTSIZE*(partnumber+1) > m_hpartfile.GetLength())
			length = (m_hpartfile.GetLength()- (PARTSIZE*partnumber));
		CreateHashFromFile(&m_hpartfile,length,hashresult);

		if (GetPartCount() > 1){
			if (memcmp(hashresult,GetPartHash(partnumber),16))
				return false;
			else
				return true;
		}
		else{
			if (memcmp(hashresult,filehash,16))
					return false;
				else
					return true;
		}
	}
}

bool CPartFile::IsCorruptedPart(uint16 partnumber){
	return corrupted_list.Find(partnumber);
}

bool CPartFile::IsMovie(){
	//TODO use filetags instead ststr
	return (strstr(GetFileName(),".avi") || strstr(GetFileName(),".mpg") || strstr(GetFileName(),".mpeg"));
}

void CPartFile::SetPriority(uint8 np){
	//UAP
	if (np == PR_AUTO)
		SetAutoPriority(true) ;
	else
	priority = np;
	//end UAP
	theApp.downloadqueue->SortByPriority();
	theApp.emuledlg->transferwnd.downloadlistctrl.UpdateItem(this);
}

void CPartFile::StopFile(){
	RemoveAllSources(true);
	paused = true;
	datarate = 0;
	transferingsrc = 0;
	theApp.emuledlg->transferwnd.downloadlistctrl.UpdateItem(this);
}

void CPartFile::PauseFile(){
	if (status==PS_COMPLETE || status==PS_COMPLETING) return;
	Packet* packet = new Packet(OP_CANCELTRANSFER,0);
	POSITION pos1,pos2;
	for( pos1 = srclist.GetHeadPosition(); ( pos2 = pos1 ) != NULL; ){
		srclist.GetNext(pos1);
		CUpDownClient* cur_src = srclist.GetAt(pos2);
		if (cur_src->GetDownloadState() == DS_DOWNLOADING){
			cur_src->socket->SendPacket(packet,false,true);
			cur_src->SetDownloadState(DS_ONQUEUE);
		}
	}
	delete packet;
	paused = true;
	datarate = 0;
	transferingsrc = 0;
	theApp.emuledlg->transferwnd.downloadlistctrl.UpdateItem(this);
}

void CPartFile::ResumeFile(){
	if (status==PS_COMPLETE || status==PS_COMPLETING) return;
	paused = false;
	lastsearchtime = 0;
	theApp.emuledlg->transferwnd.downloadlistctrl.UpdateItem(this);
}

CString CPartFile::getPartfileStatus(){ 
	CString mybuffer=""; 
	if (GetTransferingSrcCount()>0) mybuffer=GetResString(IDS_DOWNLOADING); else mybuffer=GetResString(IDS_WAITING); 
	switch (GetStatus()) {
		case PS_HASHING: 
		case PS_WAITINGFORHASH:
			mybuffer=GetResString(IDS_HASHING);
			break; 
		case PS_COMPLETING:
			mybuffer=GetResString(IDS_COMPLETING);
			break; 
		case PS_COMPLETE:
			mybuffer=GetResString(IDS_COMPLETE);
			break; 
		case PS_PAUSED:
			mybuffer=GetResString(IDS_PAUSED);
			break; 
		case PS_ERROR:
			mybuffer=GetResString(IDS_ERRORLIKE);
			break;
	} 
	return mybuffer; 
} 

sint32 CPartFile::getTimeRemaining(){ 
	if (GetDatarate()==0) return -1;
	
	return( (GetFileSize()-GetCompletedSize())/ GetDatarate());
} 

void CPartFile::PreviewFile(){
	if (!PreviewAvailable()){
		ASSERT( false );
		return;
	}
	m_bPreviewing = true;
	CPreviewThread* pThread = (CPreviewThread*) AfxBeginThread(RUNTIME_CLASS(CPreviewThread), THREAD_PRIORITY_NORMAL,0, CREATE_SUSPENDED);
	pThread->SetValues(this);
	pThread->ResumeThread();
}

bool CPartFile::PreviewAvailable(){
	uint64 space;
	space = 0;
	space = GetFreeDiskSpaceX(theApp.glob_prefs->GetTempDir());
	return !( (GetStatus() != PS_READY && GetStatus() != PS_PAUSED) 
		|| m_bPreviewing || GetPartCount() < 5 || !IsMovie() || (space + 100000000) < GetFileSize() 
		|| ( !IsComplete(0,(PARTSIZE*2)-1) || !IsComplete(PARTSIZE*(GetPartCount()-2),GetFileSize()-1)));
	
}

void CPartFile::UpdateAvailablePartsCount(){
	POSITION pos1,pos2;
	uint8 availablecounter=0;

	for (int ixPart=0;ixPart<GetPartCount();ixPart++) {
		for( pos1 = srclist.GetHeadPosition(); ( pos2 = pos1 ) != NULL; ){
				srclist.GetNext(pos1);
				if (srclist.GetAt(pos2)->IsPartAvailable(ixPart)) { availablecounter++; pos1=NULL;}
		}
	}
	if (GetPartCount()==availablecounter && availablePartsCount<GetPartCount())
		lastseencomplete=CTime::GetCurrentTime();
	availablePartsCount=availablecounter;
}

Packet*	CPartFile::CreateSrcInfoPacket(CUpDownClient* forClient){
	if (srclist.IsEmpty())
		return 0;

	CMemFile data;
	uint16 nCount = 0;

	data.Write(filehash,16);
	data.Write(&nCount,2);

	for (POSITION pos = srclist.GetHeadPosition();pos != 0;srclist.GetNext(pos)){
		CUpDownClient* cur_src = srclist.GetAt(pos);
		if (cur_src->HasLowID())
			continue;
		// only send source which have needed parts for this client if possible
		if (forClient->reqfile == this && forClient->GetPartStatus() && cur_src->GetPartStatus()){
			uint8* reqstatus = forClient->GetPartStatus();
			uint8* srcstatus = cur_src->GetPartStatus();
			bool bNeeded = false;
			for (int x = 0; x != GetPartCount(); x++){
				if (srcstatus[x] && !reqstatus[x]){
					bNeeded = true;
					break;
				}
			}
			if (!bNeeded)
				continue;
		}
		nCount++;
		uint32 dwID = cur_src->GetUserID();
		uint16 nPort = cur_src->GetUserPort();
		uint32 dwServerIP = cur_src->GetServerIP();
		uint16 nServerPort = cur_src->GetServerPort();
		data.Write(&dwID, 4);
		data.Write(&nPort, 2);
		data.Write(&dwServerIP, 4);
		data.Write(&nServerPort, 2);
		if (nCount > 500)
			break;
	}
	if (!nCount)
		return 0;
	data.Seek(16, 0);
	data.Write(&nCount, 2);

	Packet* result = new Packet(&data, OP_EMULEPROT);
	result->opcode = OP_ANSWERSOURCES;
	if (nCount > 28)
		result->PackPacket();
	return result;
}

void CPartFile::AddClientSources(CMemFile* sources){
	uint16 nCount;
	sources->Read(&nCount,2);
	for (int i = 0;i != nCount;i++){
		uint32 dwID;
		uint16 nPort;
		uint32 dwServerIP;
		uint16 nServerPort;
		sources->Read(&dwID,4);
		sources->Read(&nPort,2);
		sources->Read(&dwServerIP,4);
		sources->Read(&nServerPort,2);
		// check first if we are this source
		if (theApp.serverconnect->GetClientID() < 16777216 && theApp.serverconnect->IsConnected()){
			if ((theApp.serverconnect->GetClientID() == dwID) && theApp.serverconnect->GetCurrentServer()->GetIP() == dwServerIP)
				continue;
		}
		else if (theApp.serverconnect->GetClientID() == dwID)
			continue;
		else if (dwID < 16777216)
			continue;
		if( theApp.glob_prefs->GetMaxSourcePerFile() > this->GetSourceCount() ){
			CUpDownClient* newsource = new CUpDownClient(nPort,dwID,dwServerIP,nServerPort,this);
			theApp.downloadqueue->CheckAndAddSource(this,newsource);
		} 
	}
}
void CPartFile::UpdateUploadAutoPriority(void)
{
	if (this->IsAutoPrioritized()) {
		CPartFile* partfile = theApp.downloadqueue->GetFileByID(this->GetFileHash()) ;
		if (!partfile) return;
		if (partfile->IsPartFile()) {
			if (partfile->GetSourceCount() <= RARE_FILE)
				this->SetPriority(PR_VERYHIGH) ;
			else if (partfile->GetSourceCount() < 100)
				this->SetPriority(PR_HIGH) ;
			else
				this->SetPriority(PR_NORMAL) ;
		}
	}
}

// making this function return a higher when more sources have the extended
// protocol will force you to ask a larger variety of people for sources
int CPartFile::GetCommonFilePenalty() {
	//TODO: implement, but never return less than MINCOMMONPENALTY!
	return MINCOMMONPENALTY;
}