//this file is part of eMule
//Copyright (C)2002 Merkur ( merkur-@users.sourceforge.net / http://www.emule-project.net )
//
//This program is free software; you can redistribute it and/or
//modify it under the terms of the GNU General Public License
//as published by the Free Software Foundation; either
//version 2 of the License, or (at your option) any later version.
//
//This program is distributed in the hope that it will be useful,
//but WITHOUT ANY WARRANTY; without even the implied warranty of
//MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//GNU General Public License for more details.
//
//You should have received a copy of the GNU General Public License
//along with this program; if not, write to the Free Software
//Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.


#include "StdAfx.h"
#include "zlib/zlib.h"
#include "updownclient.h"
#include "partfile.h"
#include "opcodes.h"
#include "packets.h"
#include "emule.h"


//	members of CUpDownClient
//	which are mainly used for downloading functions 
CBarShader CUpDownClient::s_StatusBar(16);
void CUpDownClient::DrawStatusBar(CDC* dc, RECT* rect, bool onlygreyrect, bool  bFlat){ 
	COLORREF crBoth; 
	COLORREF crNeither; 
	COLORREF crClientOnly; 
	COLORREF crPending;

	if(bFlat) { 
		crBoth = RGB(0, 150, 0);
		crNeither = RGB(224, 224, 224);
		crClientOnly = RGB(0, 0, 0);
		crPending = RGB(255,255,100);
	} else { 
		crBoth = RGB(0, 192, 0);
		crNeither = RGB(240, 240, 240);
		crClientOnly = RGB(104, 104, 104);
		crPending = RGB(255, 208, 0);
	} 

	s_StatusBar.SetFileSize(reqfile->GetFileSize()); 
	s_StatusBar.SetHeight(rect->bottom - rect->top); 
	s_StatusBar.SetWidth(rect->right - rect->left); 
	s_StatusBar.Fill(crNeither); 
	if (!onlygreyrect && reqfile && m_abyPartStatus) { 
		for (uint32 i = 0;i != m_nPartCount;i++){ 
			if (m_abyPartStatus[i]){ 
				uint32 uEnd; 
				if (PARTSIZE*(i+1) > reqfile->GetFileSize()) 
					uEnd = reqfile->GetFileSize(); 
				else 
					uEnd = PARTSIZE*(i+1); 

				if (reqfile->IsComplete(PARTSIZE*i,PARTSIZE*(i+1)-1)) 
					s_StatusBar.FillRange(PARTSIZE*i, uEnd, crBoth);
				else if (m_nDownloadState == DS_DOWNLOADING && m_nLastBlockOffset < uEnd &&
				         m_nLastBlockOffset >= PARTSIZE*i)
					s_StatusBar.FillRange(PARTSIZE*i, uEnd, crPending);
				else
					s_StatusBar.FillRange(PARTSIZE*i, uEnd, crClientOnly);
			} 
		} 
	} 
	s_StatusBar.Draw(dc, rect->left, rect->top, bFlat); 
} 


bool CUpDownClient::Compare(CUpDownClient* tocomp){
	//Is this a bad check?  The possibility of two people with same hash is
	//  1 in 2^128
	//  or
	//  1 in 3.4028236692093846346337460743177e+38
	//
	//if(HasValidHash() && tocomp->HasValidHash())
	//	return memcmp(m_achUserHash, tocomp->GetUserHash(), 16);
	//
	// This wouldn't be bad if everyone went by the rules.. But they don't and
	// share ClientHashes.. So, we allow download clients with the same Hash, but
	// filter them out of the upload.. Now if some group is sharing a Hash we
	// can download from them all, but only one of them can be in our upload
	// queue at one time.

	if (HasLowID())
		return ((this->GetUserID() == tocomp->GetUserID()) && (GetServerIP() == tocomp->GetServerIP()));
	else
		return (this->GetUserID() == tocomp->GetUserID() || (this->GetIP() && this->GetIP() == tocomp->GetIP()) );
}

void CUpDownClient::AskForDownload(){
	if (theApp.listensocket->TooManySockets() && !(socket && socket->IsConnected()) ){
		if (GetDownloadState() != DS_TOOMANYCONNS)
			SetDownloadState(DS_TOOMANYCONNS);
		return;
	}
	m_bUDPPending = false;
	m_dwLastAskedTime = ::GetTickCount();
	SetDownloadState(DS_CONNECTING);
	TryToConnect();

}

bool CUpDownClient::IsSourceRequestAllowed() {
	DWORD dwTickCount = ::GetTickCount() + CONNECTION_LATENCY;
	int nTimePassedClient = dwTickCount - GetLastSrcAnswerTime();
	int nTimePassedFile   = dwTickCount - reqfile->GetLastAnsweredTime();
	bool bNeverAskedBefore = GetLastSrcReqTime() == 0;

	return (
	         //if client has the extended protocol
	         ExtProtocolAvailable() &&
	         //AND if we need more sources
	         theApp.glob_prefs->GetMaxSourcePerFileSoft() > reqfile->GetSourceCount() &&
	         //AND if...
	         (
	           //source is not complete and file is rare, allow once every 10 minutes
	           ( !m_bCompleteSource &&
	             ( reqfile->GetSourceCount() - reqfile->GetValidSourcesCount() <= RARE_FILE / 4 ||
			       reqfile->GetSourceCount() <= RARE_FILE * 2
			     ) &&
	             (bNeverAskedBefore || nTimePassedClient > SOURCECLIENTREASK)
	           ) ||
	           // otherwise, allow every 90 minutes, but only if we haven't
	           //   asked someone else in last 10 minutes
			   ( (bNeverAskedBefore || nTimePassedClient > SOURCECLIENTREASK * reqfile->GetCommonFilePenalty()) &&
		         (nTimePassedFile > SOURCECLIENTREASK)
	           )
	         )
	       );
}

void CUpDownClient::SendFileRequest(){
	ASSERT(reqfile != NULL);
	if(!reqfile)
		return;
	Packet* packet = new Packet(OP_FILEREQUEST,16);
	memcpy(packet->pBuffer,reqfile->GetFileHash(),16);
	socket->SendPacket(packet,false);
	packet->opcode = OP_SETREQFILEID;
	socket->SendPacket(packet,true);
	SetRemoteQueueRank(0);
	
	if(IsSourceRequestAllowed()) {
		OutputDebugString("OP_REQUESTSOURCES\n");
		reqfile->SetLastAnsweredTimeTimeout();
		Packet* packet = new Packet(OP_REQUESTSOURCES,16,OP_EMULEPROT);
		memcpy(packet->pBuffer,reqfile->GetFileHash(),16);
		socket->SendPacket(packet,true,true);
	}
}

void CUpDownClient::ProcessFileInfo(char* packet,uint32 size){
	CSafeMemFile* data = new CSafeMemFile((BYTE*)packet,size);
	uchar cfilehash[16];
	data->Read(cfilehash,16);
	uint16 namelength;
	data->Read(&namelength,2);
	if (m_pszClientFilename)
		delete[] m_pszClientFilename;
	m_pszClientFilename = new char[namelength+1];
	memset(m_pszClientFilename, 0, namelength+1);
	data->Read(m_pszClientFilename,namelength);	// TODO why does this overwrite the end of the buffer 
													//- because sizeof(uint16) is always 2 
	delete data;
	if ( (!reqfile) || memcmp(cfilehash,reqfile->GetFileHash(),16))
		throw CString(GetResString(IDS_ERR_WRONGFILEID)+ " (ProcessFileInfo)");
}

void CUpDownClient::ProcessFileStatus(char* packet,uint32 size){
	CSafeMemFile* data = new CSafeMemFile((BYTE*)packet,size);
	uchar cfilehash[16];
	data->Read(cfilehash,16);
	if ( (!reqfile) || memcmp(cfilehash,reqfile->GetFileHash(),16)){
		delete data;	//mf
		throw CString(GetResString(IDS_ERR_WRONGFILEID)+ " (ProcessFileStatus)");	
	}
	data->Read(&m_nPartCount,2);
	if (m_abyPartStatus)
		delete[] m_abyPartStatus;
	bool bPartsNeeded = false;
	if (!m_nPartCount){
		m_nPartCount = reqfile->GetPartCount();
		m_abyPartStatus = new uint8[m_nPartCount];
		memset(m_abyPartStatus,1,m_nPartCount);
		bPartsNeeded = true;
		m_bCompleteSource = true;
	}
	else{
		if (reqfile->GetPartCount() != m_nPartCount){
			delete data;	//mf
			throw GetResString(IDS_ERR_WRONGPARTNUMBER);
		}
		m_bCompleteSource = false;
		m_abyPartStatus = new uint8[m_nPartCount];
		uint16 done = 0;
		while (done != m_nPartCount){
			uint8 toread;
			data->Read(&toread,1);
			for (sint32 i = 0;i != 8;i++){
				m_abyPartStatus[done] = ((toread>>i)&1)? 1:0; 	
				if (m_abyPartStatus[done] && !reqfile->IsComplete(done*PARTSIZE,((done+1)*PARTSIZE)-1))
					bPartsNeeded = true;
				done++;
				if (done == m_nPartCount)
					break;
			}
		}

	}
	theApp.emuledlg->transferwnd.downloadlistctrl.UpdateItem(this);
	reqfile->UpdateAvailablePartsCount();

	if (!bPartsNeeded)
		SetDownloadState(DS_NONEEDEDPARTS);
	else if (reqfile->hashsetneeded){
		Packet* packet = new Packet(OP_HASHSETREQUEST,16);
		memcpy(packet->pBuffer,reqfile->GetFileHash(),16);
		socket->SendPacket(packet,true,true);
		SetDownloadState(DS_REQHASHSET);
		reqfile->hashsetneeded = false;
	}
	else{
		SetDownloadState(DS_ONQUEUE);
		Packet* packet = new Packet(OP_STARTUPLOADREQ,0);
		socket->SendPacket(packet,true,true);
	}
		
	reqfile->NewSrcPartsInfo();
	delete data;
}

bool CUpDownClient::AddRequestForAnotherFile(CPartFile* file){
	for (POSITION pos = m_OtherNoNeeded_list.GetHeadPosition();pos != 0;m_OtherNoNeeded_list.GetNext(pos)){
		if (m_OtherNoNeeded_list.GetAt(pos) == file)
			return false;
	}
	for (POSITION pos = m_OtherRequests_list.GetHeadPosition();pos != 0;m_OtherRequests_list.GetNext(pos)){
		if (m_OtherRequests_list.GetAt(pos) == file)
			return false;
	}
	m_OtherRequests_list.AddTail(file);
	return true;
}

void CUpDownClient::SetDownloadState(uint8 byNewState){
	if (m_nDownloadState != byNewState){
		if (m_nDownloadState == DS_DOWNLOADING ){
			m_nDownloadState = byNewState;
			for (POSITION pos = m_DownloadBlocks_list.GetHeadPosition();pos != 0;m_DownloadBlocks_list.GetNext(pos)){
				Requested_Block_Struct* cur_block = m_DownloadBlocks_list.GetAt(pos);
				reqfile->RemoveBlockFromList(cur_block->StartOffset,cur_block->EndOffset);
				delete m_DownloadBlocks_list.GetAt(pos);
			}
			m_DownloadBlocks_list.RemoveAll();

			for (POSITION pos = m_PendingBlocks_list.GetHeadPosition();pos != 0;m_PendingBlocks_list.GetNext(pos)){
				Requested_Block_Struct* cur_block = m_PendingBlocks_list.GetAt(pos)->block;
				if (reqfile)
					reqfile->RemoveBlockFromList(cur_block->StartOffset,cur_block->EndOffset);
				delete m_PendingBlocks_list.GetAt(pos)->block;
				delete m_PendingBlocks_list.GetAt(pos)->buffer;
				delete m_PendingBlocks_list.GetAt(pos);
			}
			m_PendingBlocks_list.RemoveAll();
			m_nDownDatarate = 0;
			if (byNewState == DS_NONE){
				if (m_abyPartStatus)
					delete[] m_abyPartStatus;
				m_abyPartStatus = 0;
			}
			if (socket && byNewState != DS_ERROR)
					socket->DisableDownloadLimit();
		}
		m_nDownloadState = byNewState;
		theApp.emuledlg->transferwnd.downloadlistctrl.UpdateItem(this);
	}
}

void CUpDownClient::ProcessHashSet(char* packet,uint32 size){
	if ( (!reqfile) || memcmp(packet,reqfile->GetFileHash(),16))
		throw CString(GetResString(IDS_ERR_WRONGFILEID)+" (ProcessHashSet)");	
	CSafeMemFile* data = new CSafeMemFile((BYTE*)packet,size);
	if (reqfile->LoadHashsetFromFile(data,true)){
	}
	else{
		reqfile->hashsetneeded = true;
		delete data;	//mf
		throw GetResString(IDS_ERR_BADHASHSET);
	}
	SetDownloadState(DS_ONQUEUE);
	Packet* opacket = new Packet(OP_STARTUPLOADREQ,0);
	socket->SendPacket(opacket,true,true);
	delete data;
}

void CUpDownClient::SendBlockRequests(){
	m_dwLastBlockReceived = ::GetTickCount();
	if (!reqfile)
		return;
	if (m_DownloadBlocks_list.IsEmpty()){
		uint16 count;
		Requested_Block_Struct** toadd = new Requested_Block_Struct*[3];
		if (reqfile->GetNextRequestedBlock(this,toadd,&count)){
			for (int i = 0; i != count; i++)
				m_DownloadBlocks_list.AddTail(toadd[i]);			
		}
		delete[] toadd;
	}
	while (m_PendingBlocks_list.GetCount() < 3 && !m_DownloadBlocks_list.IsEmpty()){
		Pending_Block_Struct* pblock = new Pending_Block_Struct;
		pblock->block = m_DownloadBlocks_list.RemoveHead();
		pblock->buffer = new CMemFile(10240);
		m_PendingBlocks_list.AddTail(pblock);
	}
	if (m_PendingBlocks_list.IsEmpty()){
		Packet* packet = new Packet(OP_CANCELTRANSFER,0);
		socket->SendPacket(packet,true,true);
		SetDownloadState(DS_NONEEDEDPARTS);
		return;
	}
	Packet* packet = new Packet(OP_REQUESTPARTS,40);
	CMemFile* data = new CMemFile((BYTE*)packet->pBuffer,40);
	data->Write(reqfile->GetFileHash(),16);
	POSITION pos = m_PendingBlocks_list.GetHeadPosition();
	uint32 null = 0;
	Requested_Block_Struct* block;
	for (uint32 i = 0; i != 3; i++){
		if (pos){
			block = m_PendingBlocks_list.GetAt(pos)->block;
			m_PendingBlocks_list.GetNext(pos);
			data->Write(&block->StartOffset,4);
		}
		else
			data->Write(&null,4);
	}
	pos = m_PendingBlocks_list.GetHeadPosition();
	for (uint32 i = 0; i != 3; i++){
		if (pos){
			block = m_PendingBlocks_list.GetAt(pos)->block;
			m_PendingBlocks_list.GetNext(pos);
			uint32 endpos = block->EndOffset+1;
			data->Write(&endpos,4);
		}
		else
			data->Write(&null,4);
	}
	delete data;
	socket->SendPacket(packet,true,true);
}

void CUpDownClient::ProcessBlockPacket(char* packet, uint32 size, bool packed){
	if (!(GetDownloadState() == DS_DOWNLOADING || GetDownloadState() == DS_NONEEDEDPARTS))
		return;

	theApp.UpdateReceivedBytes(size);
	m_dwLastBlockReceived = ::GetTickCount();
	CSafeMemFile* data = new CSafeMemFile((BYTE*)packet,size);
	uchar fileid[16];
	data->Read(fileid,16);
	if ( (!reqfile) || memcmp(packet,reqfile->GetFileHash(),16)){
		delete data;	//mf
		throw CString(GetResString(IDS_ERR_WRONGFILEID)+" (ProcessBlockPacket)");
	}
	uint32 nStartPos;
	uint32 nEndPos;
	uint32 nBlockSize = 0;
	data->Read(&nStartPos,4);
	if (packed){
		data->Read(&nBlockSize,4);
		nEndPos = nStartPos + (size-24);
		usedcompressiondown = true;
	}
	else
		data->Read(&nEndPos,4);
		
	if ( size != (nEndPos-nStartPos)+24){
		delete data;	//mf
		throw CString(GetResString(IDS_ERR_BADDATABLOCK)+" (ProcessBlockPacket)");
	}
	downdataratems += nEndPos-nStartPos;
	m_nTransferedDown += nEndPos-nStartPos;
	credits->AddDownloaded(nEndPos-nStartPos);
	nEndPos--;
	delete data;

	for (POSITION pos = m_PendingBlocks_list.GetHeadPosition();pos != 0;m_PendingBlocks_list.GetNext(pos)){
		Pending_Block_Struct* cur_block = m_PendingBlocks_list.GetAt(pos);
		if (cur_block->block->StartOffset <= nStartPos && cur_block->block->EndOffset >= nEndPos){
			cur_block->buffer->Write(packet+24,size-24);
			m_nLastBlockOffset = nStartPos;   // [Cax2]
			if (!packed){
				if (nEndPos == cur_block->block->EndOffset){
					if ((cur_block->block->EndOffset-cur_block->block->StartOffset)+1 != cur_block->buffer->GetLength()){
						theApp.emuledlg->AddLogLine(false,GetResString(IDS_ERR_PACKAGEERROR),reqfile->GetFileName());
					}
					else{
						char* blockbuffer = (char*)cur_block->buffer->Detach();
						reqfile->BlockReceived(cur_block->block->StartOffset,cur_block->block->EndOffset,blockbuffer);
						free(blockbuffer);
					}
					delete cur_block->block;
					delete cur_block->buffer;
					delete cur_block;
					
					m_PendingBlocks_list.RemoveAt(pos);
					SendBlockRequests();	
				}
			}
			else{
				if (nBlockSize == cur_block->buffer->GetLength()){
					BYTE* blockbuffer = cur_block->buffer->Detach();
					BYTE* unpack = new BYTE[BLOCKSIZE+300];
					uLongf unpackedsize = BLOCKSIZE+300;
					uint16 result = uncompress(unpack,&unpackedsize,blockbuffer,nBlockSize);
					if (result == Z_OK){
						reqfile->BlockReceived(cur_block->block->StartOffset,cur_block->block->StartOffset + (unpackedsize-1),(char*)unpack, nBlockSize);	
						
					}
					else{
						theApp.emuledlg->AddLogLine(false,GetResString(IDS_ERR_CORRUPTCOMPRPKG),reqfile->GetFileName(),result);
						reqfile->RemoveBlockFromList(cur_block->block->StartOffset,cur_block->block->StartOffset+1);
					}
					delete[] unpack;
					delete cur_block->block;
					delete cur_block->buffer;
					delete cur_block;
					free(blockbuffer);
					m_PendingBlocks_list.RemoveAt(pos);
					SendBlockRequests();	
				}
			}
			return;
		}
	}
}

uint32 CUpDownClient::CalculateDownloadRate(){
	m_AvarageDDR_list.AddTail(downdataratems);
	if (m_AvarageDDR_list.GetCount() > 300)
		m_AvarageDDR_list.RemoveAt(m_AvarageDDR_list.GetHeadPosition());
	
	m_nDownDatarate = 0;
	downdataratems = 0;
	for (POSITION pos = m_AvarageDDR_list.GetHeadPosition();pos != 0;m_AvarageDDR_list.GetNext(pos))
		m_nDownDatarate += m_AvarageDDR_list.GetAt(pos);

	if(m_AvarageDDR_list.GetCount() > 10)
		m_nDownDatarate /= (m_AvarageDDR_list.GetCount()/10);
	else
		m_nDownDatarate = 0;

	m_cShowDR++;
	if (m_cShowDR == 30){
		m_cShowDR = 0;
		theApp.emuledlg->transferwnd.downloadlistctrl.UpdateItem(this);
	}
	if ((::GetTickCount() - m_dwLastBlockReceived) > DOWNLOADTIMEOUT){
		Packet* packet = new Packet(OP_CANCELTRANSFER,0);
		socket->SendPacket(packet,true,true);
		SetDownloadState(DS_ONQUEUE);
	}
		
	return m_nDownDatarate;
}

uint16 CUpDownClient::GetAvailablePartCount(){
	uint16 result = 0;
	for (int i = 0;i != m_nPartCount;i++){
		if (IsPartAvailable(i))
			result++;
	}
	return result;
}

void CUpDownClient::SetRemoteQueueRank(uint16 nr){
	m_nRemoteQueueRank = nr;
	theApp.emuledlg->transferwnd.downloadlistctrl.UpdateItem(this);
}

bool CUpDownClient::SwapToAnotherFile(bool bIgnoreNoNeeded){
	if (GetDownloadState() == DS_DOWNLOADING)
		return false;
	CPartFile* SwapTo = NULL;
	if (!m_OtherRequests_list.IsEmpty()){
		for (POSITION pos = m_OtherRequests_list.GetHeadPosition();pos != 0;m_OtherRequests_list.GetNext(pos)){
			CPartFile* cur_file = m_OtherRequests_list.GetAt(pos);
			if (cur_file != reqfile && theApp.downloadqueue->IsPartFile(cur_file) && (cur_file->GetStatus(false) == PS_READY || cur_file->GetStatus(false) == PS_EMPTY) ){
				SwapTo = cur_file;
				m_OtherRequests_list.RemoveAt(pos);
				break;
			}
		}
	}
	if (!SwapTo && bIgnoreNoNeeded){
		for (POSITION pos = m_OtherNoNeeded_list.GetHeadPosition();pos != 0;m_OtherNoNeeded_list.GetNext(pos)){
			CPartFile* cur_file = m_OtherNoNeeded_list.GetAt(pos);
			if (cur_file != reqfile && theApp.downloadqueue->IsPartFile(cur_file) && (cur_file->GetStatus(false) == PS_READY || cur_file->GetStatus(false) == PS_EMPTY) ){
				SwapTo = cur_file;
				m_OtherNoNeeded_list.RemoveAt(pos);
				break;
			}
		}
	}
	if (SwapTo){
		m_OtherNoNeeded_list.AddHead(reqfile);
		theApp.downloadqueue->RemoveSource(this);
		m_nRemoteQueueRank = 0;
		if (m_abyPartStatus){
			delete[] m_abyPartStatus;
			m_abyPartStatus = 0;
		}
		m_dwLastAskedTime = 0;
		m_iRate=0;
		m_strComment="";
		theApp.downloadqueue->CheckAndAddKnownSource(SwapTo,this);
		return true;
	}
	else
		return false;

}

void CUpDownClient::UDPReaskACK(uint16 nNewQR){
	m_bUDPPending = false;
	SetRemoteQueueRank(nNewQR);
	m_dwLastAskedTime = ::GetTickCount();
	DEBUG_ONLY(theApp.emuledlg->AddLogLine(false,CString("UDP ANSWER received : %s"),GetUserName()));
}

void CUpDownClient::UDPReaskFNF(){
	m_bUDPPending = false;
	DEBUG_ONLY(theApp.emuledlg->AddLogLine(false,CString("UDP ANSWER FNF : %s"),GetUserName()));
	theApp.downloadqueue->RemoveSource(this);
	if (!socket)
		Disconnected();
}

void CUpDownClient::UDPReaskForDownload(){
	ASSERT ( reqfile );
	if(!reqfile || m_bUDPPending)
		return;

	//the line "m_bUDPPending = true;" use to be here

	if(m_byEmuleVersion >= 0x23 && m_nUDPPort != 0 && theApp.glob_prefs->GetUDPPort() != 0 &&
	   !theApp.serverconnect->IsLowID() && !HasLowID() && !(socket && socket->IsConnected())) {

		//don't use udp to ask for sources
		if(IsSourceRequestAllowed())
			return;

		DEBUG_ONLY(theApp.emuledlg->AddLogLine(false,CString("UDP REASK SEND : %s"),GetUserName()));

		m_bUDPPending = true;
		Packet* response = new Packet(OP_REASKFILEPING,16,OP_EMULEPROT);
		memcpy(response->pBuffer,reqfile->GetFileHash(),16);
		theApp.clientudp->SendPacket(response,GetIP(),GetUDPPort());
	}
}