//Copyright (C)2002 Merkur ( merkur-@users.sourceforge.net / http://emule.sf.net )
//
//This program is free software; you can redistribute it and/or
//modify it under the terms of the GNU General Public License
//as published by the Free Software Foundation; either
//version 2 of the License, or (at your option) any later version.
//
//This program is distributed in the hope that it will be useful,
//but WITHOUT ANY WARRANTY; without even the implied warranty of
//MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//GNU General Public License for more details.
//
//You should have received a copy of the GNU General Public License
//along with this program; if not, write to the Free Software
//Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

#include "StdAfx.h"
#include "partfile.h"
#include "emule.h"
#include "updownclient.h"
#include <math.h>

CPartFile::CPartFile(){
	Init();
}

CPartFile::CPartFile(CSearchFile* searchresult){
	Init();
	memcpy(filehash,searchresult->GetFileHash(),16);
	for (int i = 0; i != searchresult->taglist.GetCount();i++){
		switch (searchresult->taglist[i]->tag->specialtag){
			case FT_FILENAME:{
				filename = strdup(searchresult->taglist[i]->tag->stringvalue);
				break;
			}
			case FT_FILESIZE:{
				filesize = searchresult->taglist[i]->tag->intvalue;
				break;
			}
			default:
				CTag* newtag = new CTag(searchresult->taglist[i]->tag);
				taglist.Add(newtag);
		}
	}
	CreatePartFile();
}

CPartFile::CPartFile(CString edonkeylink){
	Init();
	try{
		if (!edonkeylink.Left(13) == CString("ed2k://|file|"))
			throw "no ed2k filelink";

		edonkeylink.Delete(0, 13);
		if (!edonkeylink.SpanExcluding("|/").GetLength())
			throw "no filename ";
		filename = strdup(edonkeylink.SpanExcluding("|").GetBuffer());
		edonkeylink.Delete(0, 1 + edonkeylink.SpanExcluding("|").GetLength());

		filesize = atol(edonkeylink.SpanExcluding("|").GetBuffer());
		edonkeylink.Delete(0, 1 + edonkeylink.SpanExcluding("|").GetLength());
		if (edonkeylink.SpanExcluding("|/").GetLength() != 32)
			throw "wrong file id";
		char idbuffer[32];
		memcpy(idbuffer,edonkeylink.SpanExcluding("|/"),32);
		char sbuffer[3];
		sbuffer[2] = 0;
		for (int i = 0;i != 16;i++){
			memcpy(sbuffer,idbuffer+i*2,2);
			uint8 value = strtol(sbuffer,0,16);
			memcpy(filehash+i,&value,1);
		}
		if (!theApp.downloadqueue->IsFileExisting(filehash))
			CreatePartFile();
		else
			status = PS_ERROR;
		}
	catch(char* error){
		char buffer[200];
		sprintf(buffer,"This ed2k link is invalid (%s)",error);
		MessageBox(0,buffer,"Invalid link",16);
		status = PS_ERROR;
	}
}

void CPartFile::Init(){
	fullname = 0;
	newdate = true;
	lastsearchtime = 0;
	paused = false;
	status = PS_EMPTY;
	transfered = 0;
	priority = PR_NORMAL;
	srcarevisible = false;
	transferingsrc = 0;
	datarate = 0;
	hashsetneeded = true;
	count = 0;
	percentcompleted = 0;
	partmetfilename = 0;
	srcpartfrequency = 0;
}

CPartFile::~CPartFile(){
	if (fullname)
		delete[] fullname;
	if (partmetfilename)
		delete[] partmetfilename;
	if (srcpartfrequency)
		delete[] srcpartfrequency;
	for (POSITION pos = gaplist.GetHeadPosition();pos != 0;gaplist.GetNext(pos))
		delete gaplist.GetAt(pos);
}

void CPartFile::CreatePartFile(){
	CFileFind ff;
	char* searchpath = new char[strlen(theApp.glob_prefs->GetTempDir())+15];
	sprintf(searchpath,"%s\\*.part",theApp.glob_prefs->GetTempDir());
	bool end = !ff.FindFile(searchpath,0);
	delete[] searchpath;
	uint16 highest = 0;
	while (!end){
		end = !ff.FindNextFile();
		if (ff.IsDirectory())
			continue;
		char* number = strdup(ff.GetFileName().GetBuffer());
		char* buffer2 = strstr(number,".");
		buffer2[0] = 0;
		if (atoi(number) > highest)
			highest = atoi(number);
		delete[] number;
	}
	ff.Close();
	partmetfilename = new char[15];
	sprintf(partmetfilename,"%i.part.met",highest+1);
	fullname = new char[strlen(theApp.glob_prefs->GetTempDir())+strlen(partmetfilename)+2];
	sprintf(fullname,"%s\\%s",theApp.glob_prefs->GetTempDir(),partmetfilename);
	char* buffer = strdup(partmetfilename);
	buffer[strlen(buffer)-4] = 0;
	CTag* partnametag = new CTag(FT_PARTFILENAME,buffer);
	delete[] buffer;
	taglist.Add(partnametag);
	
	Gap_Struct* gap = new Gap_Struct;
	gap->start = 0;
	gap->end = filesize-1;
	gaplist.AddTail(gap);

	char* partfull = strdup(fullname);
	partfull[strlen(partfull)-4] = 0;
	if (FILE* partfile = fopen(partfull,"w"))
		fclose(partfile);
	else{
		theApp.emuledlg->AddLogLine(false,"ERROR: Failed to create partfile)");
		status = PS_ERROR;
	}
	delete[] partfull;
	srcpartfrequency = new uint16[GetPartCount()];
	memset(srcpartfrequency,0,GetPartCount()*2);
	paused = false;
	SavePartFile();
}

bool CPartFile::LoadPartFile(char* in_directory,char* in_filename){
	partmetfilename = strdup(in_filename);
	directory = strdup(in_directory);
	char* buffer = new char[strlen(directory)+strlen(partmetfilename)+2];
	sprintf(buffer,"%s\\%s",directory,partmetfilename);
	fullname = buffer;
	FILE* file = 0;
	try{
		// readfile data form part.met file
		file = fopen(fullname,"rbS");
		if (!file)
			throw "Failed to open part.met file";
		uint8 version;
		fread(&version,1,1,file);
		if (version != PARTFILE_VERSION)
			throw "Invalid part.met fileversion";
		if (!this->LoadDateFromFile(file))
			throw "Failed to load part.met file-date";
		//date = 0; //debug
		if (!this->LoadHashsetFromFile(file))
			throw "Failed to load part.met file-hashset";
		uint32 tagcount;
		fread(&tagcount,4,1,file);
		for (uint32 j = 0; j != tagcount;j++){
			CTag* newtag = new CTag(file);
			switch(newtag->tag->specialtag){
				case FT_FILENAME:{
					filename = strdup(newtag->tag->stringvalue);
					delete newtag;
					break;
				}
				case FT_FILESIZE:{
					filesize = newtag->tag->intvalue;
					delete newtag;
					break;
				}
				case FT_TRANSFERED:{
					transfered = newtag->tag->intvalue;
					delete newtag;
					break;
				}
				case FT_PRIORITY:{
					priority = newtag->tag->intvalue;
					delete newtag;
					break;
				}
				case FT_STATUS:{
					paused = newtag->tag->intvalue;
					delete newtag;
					break;
				}
				default:{
					if ((!newtag->tag->specialtag) && newtag->tag->tagname[0] == FT_GAPSTART){					
						Gap_Struct* gap = new Gap_Struct; 
						gap->start = newtag->tag->intvalue;
						delete newtag;
						j++;
						CTag* newtag = new CTag(file);
						if (newtag->tag->tagname[0] != FT_GAPEND){
							delete newtag;
							throw "Invalid Gap-Tag in part.met file";
						}
						gap->end = newtag->tag->intvalue;
						// gap start = first missing byte but gap ends = first non-missing byte in edonkey
						// but I think its easier to user the real limits so:
						gap->end--;
						gaplist.AddTail(gap);
						delete newtag;
					}
					else
						taglist.Add(newtag);
				}
					
			}

		}
	}
	catch(char* error){
		if (file)
			fclose(file);
		theApp.emuledlg->AddLogLine(false,"ERROR: %s (%s => %s)",error,partmetfilename,filename);
		return false;
	}
	fclose(file);

	srcpartfrequency = new uint16[GetPartCount()];
	memset(srcpartfrequency,0,GetPartCount()*2);
	status = PS_READY;
	if (hashlist.GetCount() < GetPartCount()){
		status = PS_EMPTY;
		hashsetneeded = true;
		return true;
	}
	else {
		hashsetneeded = false;
		bool empty = true;
		for (int i = 0; i != hashlist.GetSize(); i++){
			if (IsComplete(i*PARTSIZE,((i+1)*PARTSIZE)-1))
				empty = false;
		}
		if (empty)
			status = PS_EMPTY;
	}

	if (gaplist.IsEmpty()){	// is this file complete already?
		CompleteFile(false);
		return true;
	}

	// check date of .part file - if its wrong, rehash file
	CFileFind ff;
	char* searchpath = strdup(fullname);
	searchpath[strlen(fullname)-4] = 0;
	bool end = !ff.FindFile(searchpath,0);
	delete[] searchpath;
	try{
		if (!end)
			ff.FindNextFile();
		if (end || ff.IsDirectory())
			throw ".part file not found";
		CTime lwtime;
		ff.GetLastWriteTime(lwtime);
		if (date != mktime(lwtime.GetLocalTm())){
			theApp.emuledlg->AddLogLine(false,"Warning: %s might be corrputed, performing rehash of completed parts",buffer,filename);
			// rehash
			status = PS_WAITINGFORHASH;
			CAddFileThread* addfilethread = (CAddFileThread*) AfxBeginThread(RUNTIME_CLASS(CAddFileThread), THREAD_PRIORITY_NORMAL,0, CREATE_SUSPENDED);
			addfilethread->SetValues(0,directory,ff.GetFileName().GetBuffer(),this);
			addfilethread->ResumeThread();	
		}
	}
	catch(char* error){
		ff.Close();
		theApp.emuledlg->AddLogLine(false,"ERROR: %s (%s => %s)",error,partmetfilename,filename);
		return false;
	}
	ff.Close();
	return true;
}

bool CPartFile::SavePartFile(){
	switch (status){
		case PS_WAITINGFORHASH:
		case PS_HASHING:
			return false;
	}
	FILE* file = 0;
	try{
		//get filedate
		CFileFind ff;
		char* searchpath = strdup(fullname);
		searchpath[strlen(fullname)-4] = 0;
		bool end = !ff.FindFile(searchpath,0);
		delete[] searchpath;
		if (!end)
				ff.FindNextFile();
		if (end || ff.IsDirectory())
			throw ".part file not found";
		CTime lwtime;
		ff.GetLastWriteTime(lwtime);
		date = mktime(lwtime.GetLocalTm());
		ff.Close();

		// readfile data form part.met file
		file = fopen(fullname,"wbS");
		if (!file)
			throw "Failed to open part.met file";
		//version
		uint8 version = PARTFILE_VERSION;
		fwrite(&version,1,1,file);
		//date
		fwrite(&date,4,1,file);
		//hash
		fwrite(&filehash,16,1,file);
		uint16 parts = hashlist.GetCount();
		fwrite(&parts,2,1,file);
		for (int x = 0; x != parts; x++)
			fwrite(hashlist[x],16,1,file);
		//tags
		uint32 tagcount = taglist.GetCount()+5+(gaplist.GetCount()*2);
		fwrite(&tagcount,4,1,file);
		CTag* nametag = new CTag(FT_FILENAME,filename);
		nametag->WriteTagToFile(file);
		delete nametag;
		CTag* sizetag = new CTag(FT_FILESIZE,filesize);
		sizetag->WriteTagToFile(file);
		delete sizetag;
		CTag* transtag = new CTag(FT_TRANSFERED,transfered);
		transtag->WriteTagToFile(file);
		delete transtag;
		CTag* statustag = new CTag(FT_STATUS,(paused)? 1:0);
		statustag->WriteTagToFile(file);
		delete statustag;
		CTag* prioritytag = new CTag(FT_PRIORITY,priority);
		prioritytag->WriteTagToFile(file);
		delete prioritytag;
		for (uint32 j = 0; j != taglist.GetCount();j++)
			taglist[j]->WriteTagToFile(file);
		//gaps
		char* namebuffer = new char[10];
		char* number = &namebuffer[1];
		uint16 i_pos = 0;
		for (POSITION pos = gaplist.GetHeadPosition();pos != 0;gaplist.GetNext(pos)){
			itoa(i_pos,number,10);
			namebuffer[0] = FT_GAPSTART;
			CTag* gapstarttag = new CTag(namebuffer,gaplist.GetAt(pos)->start);
			gapstarttag->WriteTagToFile(file);
			// gap start = first missing byte but gap ends = first non-missing byte in edonkey
			// but I think its easier to user the real limits
			namebuffer[0] = FT_GAPEND;
			CTag* gapendtag = new CTag(namebuffer,(gaplist.GetAt(pos)->end)+1);
			gapendtag->WriteTagToFile(file);
			delete gapstarttag;
			delete gapendtag;
			i_pos++;
		}
		delete[] namebuffer;

		if (ferror(file))
			throw "unexpected write error";
	}	
	catch(char* error){
		if (file)
			fclose(file);
		theApp.emuledlg->AddLogLine(false,"ERROR while saving partfile: %s (%s => %s)",error,partmetfilename,filename);
		return false;
	}
	fclose(file);
	return true;
}

void CPartFile::PartFileHashFinished(CKnownFile* result){
	newdate = true;
	bool errorfound = false;
	for (int i = 0; i != hashlist.GetSize(); i++){
		if (IsComplete(i*PARTSIZE,((i+1)*PARTSIZE)-1)){
			if (!(result->GetPartHash(i) && !memcmp(result->GetPartHash(i),this->GetPartHash(i),16))){
				theApp.emuledlg->AddLogLine(false,"Found corrupted part (%i) in %s",i+1,filename);		
				AddGap(i*PARTSIZE,((((i+1)*PARTSIZE)-1) >= filesize) ? filesize-1 : ((i+1)*PARTSIZE)-1);
				errorfound = true;
			}
		}
	}
	delete result;
	if (!errorfound){
		if (status == PS_COMPLETING){
			CompleteFile(true);
			return;
		}
		else
			theApp.emuledlg->AddLogLine(false,"Finished rehashing %s, file seems to be ok",filename);
	}
	else{
		status = PS_READY;
		SavePartFile();
		return;
	}
	status = PS_READY;
	SavePartFile();
	theApp.sharedfiles->SafeAddKFile(this);
}

void CPartFile::AddGap(uint32 start, uint32 end){
	POSITION pos1, pos2;
	for (pos1 = gaplist.GetHeadPosition();(pos2 = pos1) != NULL;){
		Gap_Struct* cur_gap = gaplist.GetNext(pos1);
		if (cur_gap->start >= start && cur_gap->end <= end){ // this gap is inside the new gap - delete
			gaplist.RemoveAt(pos2);
			delete cur_gap;
			continue;
		}
		else if (cur_gap->start >= start && cur_gap->start <= end){// a part of this gap is in the new gap - extend limit and delete
			end = cur_gap->end;
			gaplist.RemoveAt(pos2);
			delete cur_gap;
			continue;
		}
		else if (cur_gap->end <= end && cur_gap->end >= start){// a part of this gap is in the new gap - extend limit and delete
			start = cur_gap->start;
			gaplist.RemoveAt(pos2);
			delete cur_gap;
			continue;
		}
		else if (start >= cur_gap->start && end <= cur_gap->end){// new gap is already inside this gap - return
			return;
		}
	}
	Gap_Struct* new_gap = new Gap_Struct;
	new_gap->start = start;
	new_gap->end = end;
	gaplist.AddTail(new_gap);
	theApp.emuledlg->transferwnd.downloadlistctrl.UpdateItem(this);
	newdate = true;
}

bool CPartFile::IsComplete(uint32 start, uint32 end){
	if (end >= filesize)
		end = filesize-1;
	for (POSITION pos = gaplist.GetHeadPosition();pos != 0;gaplist.GetNext(pos)){
		Gap_Struct* cur_gap = gaplist.GetAt(pos);
		if ((cur_gap->start >= start && cur_gap->end <= end)||(cur_gap->start >= start 
			&& cur_gap->start <= end)||(cur_gap->end <= end && cur_gap->end >= start)
			||(start >= cur_gap->start && end <= cur_gap->end)){
				return false;	
			}
	}
	return true;
}

bool CPartFile::IsPureGap(uint32 start, uint32 end){
	if (end >= filesize)
		end = filesize-1;
	for (POSITION pos = gaplist.GetHeadPosition();pos != 0;gaplist.GetNext(pos)){
		Gap_Struct* cur_gap = gaplist.GetAt(pos);
		if (start >= cur_gap->start  && end <= cur_gap->end ){
			return true;
		}
	}
	return false;
}

bool CPartFile::IsAlreadyRequested(uint32 start, uint32 end){
	for (POSITION pos =  requestedblocks_list.GetHeadPosition();pos != 0; requestedblocks_list.GetNext(pos)){
		Requested_Block_Struct* cur_block =  requestedblocks_list.GetAt(pos);
		if (cur_block->StartOffset == start && cur_block->EndOffset == end)
			return true;
	}
	return false;
}

bool CPartFile::GetNextEmptyBlockInPart(uint16 partnumber,Requested_Block_Struct* result){
	for (uint32 i = 0; i != ceil((float)PARTSIZE/BLOCKSIZE);i++){
		uint32 start = (PARTSIZE*partnumber) + i*BLOCKSIZE;
		uint32 end	 = (PARTSIZE*partnumber) + ((i+1)*BLOCKSIZE)-1;
		if (end >= PARTSIZE*(partnumber+1))
			end = (PARTSIZE*(partnumber+1))-1;
		if (start >= GetFileSize())
			break;
		if (end >= GetFileSize())
			end = GetFileSize()-1;
		if ( (!IsComplete(start,end)) && !IsAlreadyRequested(start,end)){
			if (result){
				result->StartOffset = start;
				result->EndOffset = end;
				memcpy(result->FileID,GetFileHash(),16);
			}
			return true;
		}
	}
	return false;
}

void CPartFile::FillGap(uint32 start, uint32 end){
	POSITION pos1, pos2;
	for (pos1 = gaplist.GetHeadPosition();(pos2 = pos1) != NULL;){
		Gap_Struct* cur_gap = gaplist.GetNext(pos1);
		if (cur_gap->start >= start && cur_gap->end <= end){ // our part fills this gap completly
			gaplist.RemoveAt(pos2);
			delete cur_gap;
			continue;
		}
		else if (cur_gap->start >= start && cur_gap->start <= end){// a part of this gap is in the part - set limit
			cur_gap->start = end+1;
		}
		else if (cur_gap->end <= end && cur_gap->end >= start){// a part of this gap is in the part - set limit
			cur_gap->end = start-1;
		}
		else if (start >= cur_gap->start && end <= cur_gap->end){
			uint32 buffer = cur_gap->end;
			cur_gap->end = start-1;
			cur_gap = new Gap_Struct;
			cur_gap->start = end+1;
			cur_gap->end = buffer;
			gaplist.InsertAfter(pos1,cur_gap);
			return;
		}
	}
	theApp.emuledlg->transferwnd.downloadlistctrl.UpdateItem(this);
	newdate = true;
	SavePartFile();
}

void CPartFile::DrawStatusBar(CDC* dc, RECT* rect){
	CBrush br(RGB(0,0,0));
	RECT gaprect;
	// grey rect
	dc->FillRect(rect,&CBrush(RGB(220,220,220)));
	gaprect.top = rect->top + 4;
	gaprect.bottom = rect->bottom;
	gaprect.left = rect->left;
	gaprect.right = rect->right;
	// black rect
	if (status != PS_COMPLETE)
		dc->FillRect(&gaprect,&br);
	else
		dc->FillRect(&gaprect,&CBrush(RGB(0,150,0)));
	uint32 allgaps = 0;
	float blockpixel = (float)(rect->right - rect->left)/((float)filesize/1024);
	// red gaps
	for (POSITION pos = gaplist.GetHeadPosition();pos != 0;gaplist.GetNext(pos)){
		Gap_Struct* cur_gap = gaplist.GetAt(pos);
		bool gapdone = false;
		uint32 gapstart = cur_gap->start;
		uint32 gapend = cur_gap->end;
		for (int i = 0; i != GetPartCount(); i++){
			if (gapstart >= i*PARTSIZE && gapstart <= (i+1)*PARTSIZE){ // is in this part?
				if (gapend <= (i+1)*PARTSIZE)
					gapdone = true;
				else{
					gapend = (i+1)*PARTSIZE; // and next part
				}
				// paint
				COLORREF color;
				if (srcpartfrequency && srcpartfrequency[i]) // frequency?
					color = RGB(0,
						(210-(22*(srcpartfrequency[i]-1)) < 0)? 0:210-(22*(srcpartfrequency[i]-1))
						,255);
				else
					color = RGB(255,0,0);
				gaprect.right = rect->left + (uint32)((float)((float)gapstart/1024)*blockpixel);

				gaprect.left = rect->left +  (uint32)((float)((float)gapend/1024)*blockpixel);
				dc->FillRect(&gaprect,&CBrush(color));
				
				if (gapdone) // finished?
					break;
				else{
					gapstart = gapend;
					gapend = cur_gap->end;
				}
			}
		}
		allgaps += cur_gap->end - cur_gap->start;
	}
	// yellow pending parts
	for (POSITION pos = requestedblocks_list.GetHeadPosition();pos != 0;requestedblocks_list.GetNext(pos)){
		Requested_Block_Struct* block = requestedblocks_list.GetAt(pos);
		gaprect.right = rect->left + (uint32)((float)((float)block->StartOffset/1024)*blockpixel);
		gaprect.left = rect->left +  (uint32)((float)((float)block->EndOffset/1024)*blockpixel);
		dc->FillRect(&gaprect,&CBrush(RGB(255,255,100)));
	}
	// green progress
	gaprect.bottom = gaprect.top;
	gaprect.top -= 4;
	gaprect.left = rect->left;
	gaprect.right = rect->left+ (uint32)((float)((float)((filesize-allgaps)+1)/1024)*blockpixel);
	dc->FillRect(&gaprect,&CBrush(RGB(0,150,0)));

	// for sorting
	percentcompleted = (uint8)((float)((float)allgaps/filesize) * 100);
}

void CPartFile::WritePartStatus(CFile* file){
	uint16 parts = hashlist.GetCount();
	file->Write(&parts,2);
	uint16 done = 0;
	while (done != parts){
		uint8 towrite = 0;
		for (uint32 i = 0;i != 8;i++){
			if (IsComplete(done*PARTSIZE,((done+1)*PARTSIZE)-1))
				towrite |= (1<<i);
			done++;
			if (done == parts)
				break;
		}
		file->Write(&towrite,1);
	}
}

uint8 CPartFile::GetStatus(){
	if ((!paused) || status == PS_ERROR)
		return status;
	else
		return PS_PAUSED;
}

uint32 CPartFile::Process(sint16 reducedownload/*in percent*/){
	// check if we want new sources from server
	if (!theApp.listensocket->TooManySockets()){//can't do anything when we have to mayne connections
	
		if (( (!lastsearchtime) || (::GetTickCount() - lastsearchtime) > SERVERREASKTIME) && theApp.serverconnect->IsConnected()){
			//local server
			lastsearchtime = ::GetTickCount();
			Packet* packet = new Packet(OP_GETSOURCES,16);
			memcpy(packet->pBuffer,filehash,16);
			theApp.serverconnect->SendPacket(packet,true);
		}
		POSITION pos1,pos2;
		for (POSITION pos1 = srclist.GetHeadPosition();( pos2 = pos1 ) != NULL;){
			srclist.GetNext(pos1);
			CUpDownClient* cur_src = srclist.GetAt(pos2);
			// now check if we should (re)ask our sources
			if (theApp.serverconnect->IsConnected() && ((!cur_src->GetLastAskedTime()) || (::GetTickCount() - cur_src->GetLastAskedTime()) > FILEREASKTIME)){
				switch (cur_src->GetDownloadState()){
					case DS_DOWNLOADING:	
					case DS_BANNED:
					case DS_ERROR:
						break;
					case DS_LOWTOLOWIP:	// if we now have a high ip we can ask
						if (theApp.serverconnect->IsLowID())
							break;
					case DS_NONEEDEDPARTS:
					case DS_CONNECTING:
					case DS_ONQUEUE:
					case DS_NONE:
					case DS_WAITCALLBACK:
						cur_src->AskForDownload();
				}
			}
		}
	}

	// calculate datarate, set limit etc.
	transferingsrc = 0;
	datarate = 0;
	POSITION pos1,pos2;
	for( pos1 = srclist.GetHeadPosition(); ( pos2 = pos1 ) != NULL; ){
		srclist.GetNext(pos1);
		CUpDownClient* cur_src = srclist.GetAt(pos2);
		if (cur_src->GetDownloadState() == DS_DOWNLOADING){
			transferingsrc++;
			uint32 cur_datarate = cur_src->CalculateDownloadRate();
			datarate += cur_datarate;
			if (reducedownload && cur_src->GetDownloadState() == DS_DOWNLOADING){
				uint32 limit = (uint32)(((float)reducedownload/100)*cur_datarate)/10;		
					if (limit < 1000 && reducedownload == 200)
						limit += 1000;
					else if (limit < 1)
						limit = 1;
					cur_src->socket->SetDownloadLimit(limit);
			}
			else
				cur_src->socket->DisableDownloadLimit();

		}
	}

	count++;
	if (count == 30){
		count = 0;
		theApp.emuledlg->transferwnd.downloadlistctrl.UpdateItem(this);
	}
	return datarate;
}

void CPartFile::AddSources(CMemFile* sources,uint32 serverip, uint16 serverport){
	uint8 count;
	sources->Read(&count,1);
	for (int i = 0;i != count;i++){
		uint32 userid;
		sources->Read(&userid,4);
		uint16 port;
		sources->Read(&port,2);
		// check first if we are this source
		if (theApp.serverconnect->GetClientID() < 16777216 && theApp.serverconnect->IsConnected()){
			if ((theApp.serverconnect->GetClientID() == userid) && inet_addr(theApp.serverconnect->GetCurrentServer()->GetFullIP()) == serverip)
				continue;
		}
		else if (theApp.serverconnect->GetClientID() == userid)
			continue;

		CUpDownClient* newsource = new CUpDownClient(port,userid,serverip,serverport,this);
		
		// now check if we have already this source and if not add it
		theApp.downloadqueue->CheckAndAddSource(this,newsource);
	}
}

void CPartFile::NewSrcPartsInfo(){
	delete[] srcpartfrequency;
	srcpartfrequency = new uint16[GetPartCount()];
	memset(srcpartfrequency,0,GetPartCount()*2);
	for (POSITION pos = srclist.GetHeadPosition();pos != 0;srclist.GetNext(pos)){
		for (int i = 0; i != GetPartCount(); i++){
			if (srclist.GetAt(pos)->IsPartAvailable(i))
				srcpartfrequency[i] +=1;
		}
	}
	theApp.emuledlg->transferwnd.downloadlistctrl.UpdateItem(this);
}

bool CPartFile::GetNextRequestedBlock(CUpDownClient* sender,Requested_Block_Struct** newblocks,uint16* count){
#define	BLOCKCOUNT		3
#define ROUND(x) (floor((float)x+0.5f))
	uint16 newblockcount = 0;
	uint8* partsav = sender->GetPartStatus();
	*count = 0;
	bool finished = false;
	while (!finished){
		uint16 randomness = (uint16)ROUND(((float)rand()/RAND_MAX)*(GetPartCount()-1));
		uint16 possiblepart = 0xFFFF;
		uint16 goodpart = 0xFFFF;
		
		for (uint16 i = 0;i != GetPartCount();i++){
			if (partsav[randomness] && !IsComplete(randomness*PARTSIZE,((randomness+1)*PARTSIZE)-1)){
				if (IsCorruptedPart(randomness)){
					if (GetNextEmptyBlockInPart(randomness,0)){
						goodpart = randomness;
						break;
					}
				}
				else if (IsMovie() && (randomness <= 1 || randomness >= GetPartCount()-2)){
					// if this is a movie we want the last und first part first to make a preview possible
					if (GetNextEmptyBlockInPart(randomness,0)){
							goodpart = randomness;
							break;
					}
				}
				else if (IsPureGap(randomness*PARTSIZE,((randomness+1)*PARTSIZE)-1)){
					if (GetNextEmptyBlockInPart(randomness,0))
						possiblepart = randomness;
				}
				else{
					if (GetNextEmptyBlockInPart(randomness,0)){
						goodpart = randomness;
					}
				}
			}
			randomness++;
			if (randomness == GetPartCount())
				randomness = 0;
		}

		uint16 usedpart;
		if (goodpart != 0xFFFF)
			usedpart = goodpart;
		else if (possiblepart != 0xFFFF)
			usedpart = possiblepart;
		else{
			if (!newblockcount){
				return false;
			}
			else
				break;
		}
		while (true){
			Requested_Block_Struct* block = new Requested_Block_Struct;
			if (GetNextEmptyBlockInPart(usedpart,block)){
				requestedblocks_list.AddTail(block);
				newblocks[newblockcount] = block;
				newblockcount++;
				if (newblockcount == BLOCKCOUNT){
					finished = true;
					break;
				}
			}
			else
				break;
		}

	} //wend
	*count = newblockcount;
	return true;
}

void  CPartFile::RemoveBlockFromList(uint32 start,uint32 end){
	POSITION pos1,pos2;
	for (pos1 = requestedblocks_list.GetHeadPosition(); ( pos2 = pos1 ) != NULL; ){
	   requestedblocks_list.GetNext(pos1);
		if (requestedblocks_list.GetAt(pos2)->StartOffset <= start && requestedblocks_list.GetAt(pos2)->EndOffset >= end)
			requestedblocks_list.RemoveAt(pos2);
	}
}

void  CPartFile::BlockReceived(uint32 start,uint32 end, char* blockdata,uint32 transferedsize){
	// TODO buffer blocks to decrease hd access
	RemoveBlockFromList(start,end);
	if (IsComplete(start,end))
		return;
	char* partfilename = strdup(fullname);
	partfilename[strlen(fullname)-4] = 0;
	CFile* file = 0;
	try{
		file = new CFile(partfilename,CFile::modeWrite|CFile::shareDenyNone);
		delete[] partfilename;
		if (file->GetLength() < end+1)
			file->SetLength(end+1);
		file->Seek(start,0);
		file->Write(blockdata,(end-start)+1);
		uint16 partnumber = (uint16)floor((float)start/PARTSIZE);		
		uint32 length = PARTSIZE;
		if (PARTSIZE*(partnumber+1) > file->GetLength())
			length = (file->GetLength()- (PARTSIZE*partnumber));		
		file->Close();
		delete file;
		FillGap(start,end);
		if (transferedsize)
			transfered += transferedsize;
		else
			transfered += (end-start)+1;

		// if this part is complete or was coruupted check the hash

	
		if (IsComplete(PARTSIZE*partnumber,(PARTSIZE*(partnumber+1))-1)){
			if (!HashSinglePart(partnumber)){
				theApp.emuledlg->AddLogLine(true,"Downloaded part %i is corrupted :(  (%s)",partnumber,GetFileName());
				AddGap(PARTSIZE*partnumber,(PARTSIZE*partnumber+length)-1);
				corrupted_list.AddTail(partnumber);
			}
			else{
				if (status == PS_EMPTY){
					status = PS_READY;
					theApp.sharedfiles->SafeAddKFile(this);
				}
			}
		}
		else if (IsCorruptedPart(partnumber) && theApp.glob_prefs->IsICHEnabled()){
			if (HashSinglePart(partnumber)){
				FillGap(PARTSIZE*partnumber,(PARTSIZE*partnumber+length)-1);
				RemoveBlockFromList(PARTSIZE*partnumber,(PARTSIZE*partnumber+length)-1);
				theApp.emuledlg->AddLogLine(true,"ICH: Recovered corrupted part %i  (%s)",partnumber,GetFileName());
			}
		}
		
		if (gaplist.IsEmpty())		//file complete
			CompleteFile(false);	
	}
	catch(CFileException* error){
		Beep(800,400);
		if (file){
			file->Close();
			delete file;
		}
		if (error->m_cause == CFileException::diskFull)
			theApp.emuledlg->AddLogLine(true,"You have insufficient diskspace to download %s!",this->GetFileName());
		else{
			char buffer[150];
			error->GetErrorMessage(buffer,150);
			theApp.emuledlg->AddLogLine(true,"Unexpected file error while writing %s : %s",GetFileName(),buffer);
		}

		paused = true;
		status = PS_ERROR;
	}
}

void CPartFile::CompleteFile(bool hashingdone){
	status = PS_COMPLETING;
	if (!hashingdone){
		datarate = 0;
		char* partfileb = strdup(partmetfilename);
		partfileb[strlen(partmetfilename)-4] = 0;
		CAddFileThread* addfilethread = (CAddFileThread*) AfxBeginThread(RUNTIME_CLASS(CAddFileThread), THREAD_PRIORITY_BELOW_NORMAL,0, CREATE_SUSPENDED);
		addfilethread->SetValues(0,theApp.glob_prefs->GetTempDir(),partfileb,this);
		addfilethread->ResumeThread();	
		delete[] partfileb;
		return;
	}
	else{
		StopFile();
		char* partfilename = strdup(fullname);
		partfilename[strlen(fullname)-4] = 0;
		CFile* file = 0;
		char* newname = new char[strlen(GetFileName())+strlen(theApp.glob_prefs->GetIncomingDir())+10];
		sprintf(newname,"%s\\%s",theApp.glob_prefs->GetIncomingDir(),GetFileName());
		if (rename(partfilename,newname)){
			delete[] partfilename;
			delete[] newname;
			theApp.emuledlg->AddLogLine(true,"Unexpected file error while completing %s. File paused, restart eMule to start another completing-try",GetFileName());
			paused = true;
			status = PS_ERROR;
			return;
		}
		if (remove(fullname))
			theApp.emuledlg->AddLogLine(true,"Failed to delete %s, you will need to do this by hand",fullname);
		delete[] partfilename;
		delete[] fullname;
		fullname = newname;
		delete[] directory;
		directory = strdup(theApp.glob_prefs->GetIncomingDir());
		status = PS_COMPLETE;
		paused = false;
		theApp.emuledlg->AddLogLine(true,"Finished downloading %s :-)",GetFileName());
		theApp.knownfiles->SafeAddKFile(this);
		theApp.downloadqueue->RemoveFile(this);
		theApp.emuledlg->transferwnd.downloadlistctrl.UpdateItem(this);
	}
}

void  CPartFile::RemoveAllSources(){
	//TODO transfer sources to other downloading files if possible
	POSITION pos1,pos2;
	for( pos1 = srclist.GetHeadPosition(); ( pos2 = pos1 ) != NULL; ){
		srclist.GetNext(pos1);
		theApp.downloadqueue->RemoveSource(srclist.GetAt(pos2));
	}
}

void CPartFile::DeleteFile(){
	RemoveAllSources();
	if (remove(fullname))
		theApp.emuledlg->AddLogLine(true,"Failed to delete %s",fullname);
	char* partfilename = strdup(fullname);
	partfilename[strlen(fullname)-4] = 0;
	if (remove(partfilename))
		theApp.emuledlg->AddLogLine(true,"Failed to delete %s",partfilename);
	delete[] partfilename;
	theApp.downloadqueue->RemoveFile(this);
	theApp.sharedfiles->RemoveFile(this);
	theApp.emuledlg->transferwnd.downloadlistctrl.RemoveFile(this);
	delete this;
}

bool CPartFile::HashSinglePart(uint16 partnumber){
	char* partfilename = strdup(fullname);
	partfilename[strlen(fullname)-4] = 0;
	if ((GetHashCount() <= partnumber) && (GetPartCount() > 1)){
		theApp.emuledlg->AddLogLine(true,"Warning: Unable to hash downloaded part - hashset incomplete (%s)",GetFileName());
		return true;
	}
	else if(!GetPartHash(partnumber)){
		theApp.emuledlg->AddLogLine(true,"Error: Unable to hash downloaded part - hashset incomplete (%s). This should never happen",GetFileName());
		return true;		
	}
	else{
		CFile* file = new CFile(partfilename,CFile::modeRead|CFile::shareDenyNone);
		delete[] partfilename;
		uchar hashresult[16];
		file->Seek(PARTSIZE*partnumber,0);
		uint32 length = PARTSIZE;
		if (PARTSIZE*(partnumber+1) > file->GetLength())
			length = (file->GetLength()- (PARTSIZE*partnumber));
		CreateHashFromFile(file,length,hashresult);
		file->Close();
		delete file;
		if (GetPartCount() > 1){
			if (memcmp(hashresult,GetPartHash(partnumber),16))
				return false;
			else
				return true;
		}
		else{
			if (memcmp(hashresult,filehash,16))
					return false;
				else
					return true;
		}
	}
}

bool CPartFile::IsCorruptedPart(uint16 partnumber){
	return corrupted_list.Find(partnumber);
}

bool CPartFile::IsMovie(){
	//TODO use filetags instead ststr
	return (strstr(GetFileName(),".avi") || strstr(GetFileName(),".mpg") || strstr(GetFileName(),".mpeg"));
}

void CPartFile::SetPriority(uint8 np){
	priority = np;
	theApp.downloadqueue->SortByPriority();
	theApp.emuledlg->transferwnd.downloadlistctrl.UpdateItem(this);
}

void CPartFile::StopFile(){
	RemoveAllSources();
	paused = true;
	datarate = 0;
	transferingsrc = 0;
	theApp.emuledlg->transferwnd.downloadlistctrl.UpdateItem(this);
}

void CPartFile::PauseFile(){
	Packet* packet = new Packet(OP_CANCELTRANSFER,0);
	POSITION pos1,pos2;
	for( pos1 = srclist.GetHeadPosition(); ( pos2 = pos1 ) != NULL; ){
		srclist.GetNext(pos1);
		CUpDownClient* cur_src = srclist.GetAt(pos2);
		if (cur_src->GetDownloadState() == DS_DOWNLOADING){
			cur_src->socket->SendPacket(packet,false,true);
			cur_src->SetDownloadState(DS_ONQUEUE);
		}
	}	
	paused = true;
	datarate = 0;
	transferingsrc = 0;
	theApp.emuledlg->transferwnd.downloadlistctrl.UpdateItem(this);
}

void CPartFile::ResumeFile(){
	paused = false;
	lastsearchtime = 0;
	theApp.emuledlg->transferwnd.downloadlistctrl.UpdateItem(this);
}

